﻿<#
  .SYNOPSIS
   Script to monitor SQL Error Log records on multiple SQL Instances from a central management server.

  .DESCRIPTION
   This script can be used to report on unusual entries in the SQL error log file(s) and to report on failed logins. 
   
  .EXAMPLE
    cd D:\Monitoring
    .\Monitor_SQL_Errorlog.ps1
    
  .NOTES
    Requirements | Dependencies
    - This script requires an input file with SQL Instance names, whose error log files need to be monitored. Format: 
        - <servern name>\<instance name> for a SQL named instnace
        - <server name> for a SQL default instance
    - This script uses a stored procedure to filter the error logs collected and remove entries that do not require further investigation:
        - 'Cleanup_Errorlog_Holding_Table'
    - 'Cleanup_Errorlog_Holding_Table' requires a table with strings (parts of error log messages) that can be ignored.
    - This script uses a number of stored procedures to send out email alerts.  
      These stored procedures use msdb.dbo.sp_send_dbmail and require a valid @profile_name and @recipient mail address:
      - Mail_Failed_Logins 
      - Mail_SQLErrorlog
      - Mail_Warning_FailedLogins
      - Mail_Warning_SQLErrorlog
      
    2021/02 Willem Gossink
#>


### =====  USER DEFINED PARAMETERS  ===== ###

    # Path + name of file with SQL Instances whose SQL errorlog files you want to monitor.
$SQLInstanceFile = "D:\SQLErrorlogMonitor\SQLInstances.txt"
    # Retain history for xx days
$RetainHistoryDays = 28   
    # Server where holding table lives
$SQLManagementInstance = "SQL001"
    # Database where holding table lives
$SQLManagementDB = "SQLErrorLogMonitor"
    # Database Mail Profile Name on $SQLManagementInstance
$SQLMailProfileName = '<MailProfileName>'
    # Mail recpients; separate with semicolons where applicable
$SQLMailRecipients = '<mail_recipient1>;<mail_recipient2>'


### =====  START WORK  ===== ###

# Set up holding table and table to track progress + clean up holding table
$SQL = "IF OBJECT_ID('ProgressMonitor')        IS NULL CREATE TABLE ProgressMonitor ([SQLInstance] SYSNAME, [LastUpdate] DATETIME) ;
        IF OBJECT_ID('tempdb.dbo.SQLErrorLog') IS NULL CREATE TABLE tempdb.dbo.SQLErrorLog ([SQLInstance] SYSNAME, [LogDate] DATETIME, [ProcessInfo] VARCHAR(50), [Text] VARCHAR(1000), [Count] INT) ;
        IF OBJECT_ID('SQLErrorLog')            IS NULL CREATE TABLE            SQLErrorLog ([SQLInstance] SYSNAME, [LogDate] DATETIME, [ProcessInfo] VARCHAR(50), [Text] VARCHAR(1000), [Count] INT) ;
        TRUNCATE TABLE tempdb.dbo.SQLErrorLog ;
        DELETE SQLErrorLog WHERE [LogDate] < '" + $((Get-Date).AddDays(-$RetainHistoryDays)) + "' ;

"
Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL

# For each SQL instance, retrieve new error log records
$InstanceList = Import-Csv -Header SQLInstance $SQLInstanceFile
ForEach ($Instance in $InstanceList | where {$_.SQLInstance -ne ''}) {

    $SQLInstance = $Instance.SQLInstance
      # Create new ProgressMonitor record in case of new SQL Instance. Determine date/time of last errorlog record retrieved for the current SQL Instance. 
    $SQL = "
      IF NOT EXISTS (SELECT 1 FROM ProgressMonitor WHERE [SQLInstance] = '$SQLInstance')
       INSERT ProgressMonitor SELECT '$SQLInstance', GETDATE()-1  ;

      -- Show date/time of last errorlog line recorded in database (datetime conversion to char because PS strips off the ms part of a datetime field)
      SELECT CONVERT(CHAR(23),[LastUpdate],25) AS 'Date' 
      FROM   ProgressMonitor 
      WHERE  [SQLInstance] = '$SQLInstance' ;
    "
    $LastLineMonitored = Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL  

      # On each monitored SQL instance, retrieve errror log entries starting from the date/time in $LastLineMonitored
    $SQL = "
      DECLARE @LastLineMonitored DATETIME,
              @FirstLineStored   DATETIME,
              @LastLineStored    DATETIME,
              @CurErrLog         TINYINT,
              @MaxErrLog         TINYINT

      SELECT  @LastLineMonitored = '" + $LastLineMonitored.Date + "'

      DECLARE @TempErrLog TABLE ([Servername] SYSNAME DEFAULT @@SERVERNAME,[LogDate] DATETIME, [ProcessInfo] VARCHAR(50), [Text] VARCHAR(1000))
      INSERT INTO @TempErrLog ([LogDate], [ProcessInfo], [Text])
        EXEC master.SYS.XP_READERRORLOG 0, 1, NULL, NULL, @LastLineMonitored ; 
  
      SELECT @FirstLineStored = MIN(LogDate) FROM @TempErrLog ;

      IF @FirstLineStored > @LastLineMonitored  -- log may have been reinitialized; retrieve more lines from older versions of the errorlog
      BEGIN
        DECLARE @ErrLogCount TABLE ([ArchiveNumber] TINYINT, [Date] DATETIME, [Size] BIGINT)
        INSERT  @ErrLogCount EXEC SP_ENUMERRORLOGS
        SELECT  @MaxErrLog = MAX([ArchiveNumber]) FROM @ErrLogCount
        SELECT  @CurErrLog = 1
  
        WHILE (@LastLineMonitored < @FirstLineStored AND  @CurErrLog <= @MaxErrLog)
        BEGIN
          INSERT INTO @TempErrLog ([LogDate], [ProcessInfo], [Text])
            EXEC master.SYS.XP_READERRORLOG @CurErrLog, 1, NULL, NULL, @LastLineMonitored
          SELECT @FirstLineStored = MIN(LogDate) FROM @TempErrLog
          SELECT @CurErrLog += 1
        END 
      END

      -- Build command to insert the error log lines into a holding table. Records are grouped, so each distinct occurrence is listed once, with a 'count'.
      SELECT ' 
         SELECT 
           '''+ [Servername] + ''',
           '''+ CONVERT(CHAR(23),MAX([LogDate]),25) + ''',
           '''+ [ProcessInfo] + ''' ,
           '''+ REPLACE([Text], '''', '''''') + ''' , -- replace single quotes in [Text] field by double single quotes
           '''+ CONVERT(VARCHAR(10),COUNT(*)) + '''  
         UNION ALL
      '
      FROM @TempErrLog
      GROUP BY [Servername], [ProcessInfo], [Text]
    "

      # Store all errrorlog records in an array. The SQL query formats the entries in a 'select <errorlog fields> union all' format that allows the records to be inserted into a sql table
    $Result = Invoke-Sqlcmd -ServerInstance $SQLInstance -Query $SQL 

      # Determine the number of elements in the array. Then, strip off the 'UNION ALL' statement from the last element only.
    $ComposedResult = ''
    $RecordCount = $($Result.Column1).Count
    If ($RecordCount -eq 1) {
      $Body = ''
      $Tail = $Result.Column1 -replace 'UNION ALL', ''
    }
    Elseif ($RecordCount -gt 1) {
      $Body = $Result.Column1[0..$($RecordCount-2)]
      $Tail = $Result.Column1[$($RecordCount-1)] -replace 'UNION ALL', ''
    }
      # Concatenate the error log entries, with the final 'UNION ALL'  removed
    $ComposedResult= $Body + $Tail 

      # If any error log records have been recorded, insert these to the holding table in tempdb and update the ProgressMonitor table
    if ($ComposedResult) {
        # Insert all errorlog records into the holding table in tempdb on the management server
      $SQL= "INSERT tempdb.dbo.SQLErrorLog " + $ComposedResult
      Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL
  
        # Update ProgressMonitor table to reflect last errorlog record stored
      $SQL= "
        UPDATE ProgressMonitor SET [LastUpdate] = (SELECT MAX(LogDate) FROM tempdb.dbo.SQLErrorLog WHERE [SQLInstance] = '$SQLInstance')
        WHERE [SQLInstance] = '$SQLInstance'
      "
      Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL
    }
}

# Remove entries you want to ignore from the tempdb holding table (these 'standard' exclusions live in a separate SQL table in $SQLManagementDB)
# Determine whether mails need to be sent
$SQL = "
    SET NOCOUNT ON ;
    EXEC Cleanup_Errorlog_Holding_Table ;
    SELECT COUNT(*) AS 'LogRecords' FROM tempdb.dbo.SQLErrorLog
"
$FinalResult = Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL

# Send applicable mail(s) and move remaining records to holding table
if ($FinalResult.LogRecords -gt 0) {
    $SQL = "
        EXEC Process_SQLErrorlogResults 
            @MailRecipients  = '$SQLMailRecipients', 
            @MailProfileName = '$SQLMailProfileName' ; 
        INSERT [SQLErrorLog] 
           SELECT * FROM tempdb.dbo.SQLErrorLog ;
    "
 Invoke-Sqlcmd -ServerInstance $SQLManagementInstance -Database $SQLManagementDB -Query $SQL
}
