USE [SQLErrorLogMonitor]
GO

IF (OBJECT_ID('Cleanup_Errorlog_Holding_Table')) IS NULL
  EXEC ('CREATE PROC [Cleanup_Errorlog_Holding_Table] AS PRINT ''STUB''')
GO


ALTER PROCEDURE [dbo].[Cleanup_Errorlog_Holding_Table]
AS
/*** Script to clean up the SQL Errorlog holding table, based on a table containing entries to be excluded.
     The exclude table contains entries that need not be reported, so these can be deleted from the
     holding table. After this 'cleanup', all remaining entries in the holding table will be mailed as a
     notification.
     An exclusion MUST contain
     - a text string. Records in the errorlog holding table containing this text 
       (+ any other text surrounding it) will be deleted.
     - in the text column: a percentage sign as first and as last character (unless it represents the entire text of the line you want deleted).
     An exclusion MAY contain
     - a 'starthour' and 'endhour'. All entries recorded between starthour and endhour 
       and containing the specified text will be deleted.
     - a server name. All entries containing the specified text, coming from this server
       and recorded between starthour and endhour will be deleted.
     - in the text column: a percentage sign used as wildcard within a text string, for pieces of text that are variable, such as numbers or names
*/
SET NOCOUNT ON

DECLARE cleanup CURSOR FOR
SELECT   RTRIM(LTRIM([text]))
       , SQLInstance
       , StartHour
       , EndHour 
FROM     dbo.SQLErrorLog_Exclude

DECLARE   @text        VARCHAR(200)
        , @sqlinstance SYSNAME
        , @starthour   SMALLINT
        , @endhour     SMALLINT
OPEN cleanup
FETCH NEXT FROM cleanup INTO @text, @sqlinstance, @starthour, @endhour
WHILE @@FETCH_STATUS = 0
  BEGIN
    IF @starthour IS NULL AND @endhour IS NULL
      -- Delete all records with this text, regardless of time reported
      DELETE tempdb.dbo.SQLErrorLog 
      WHERE PATINDEX(@text,[Text]) > 0 
       AND ([SQLInstance] = @sqlinstance 
            OR @sqlinstance IS NULL)
    ELSE IF (@starthour = @endhour) OR (@starthour IS NULL AND @endhour IS NOT NULL)
     -- Delete records on the basis of @endhour
      DELETE tempdb.dbo.SQLErrorLog
      WHERE PATINDEX(@text,[Text]) > 0 
       AND ([SQLInstance] = @sqlinstance 
            OR @sqlinstance IS NULL)
       AND DATEPART(hh, [LogDate]) = @endhour
    ELSE IF (@starthour IS NOT NULL 
             AND @endhour IS NULL)
     -- Delete records on the basis of @starthour
      DELETE tempdb.dbo.SQLErrorLog
      WHERE PATINDEX(@text,[Text]) > 0 
       AND ([SQLInstance] = @sqlinstance 
            OR @sqlinstance IS NULL)
       AND DATEPART(hh, [LogDate]) = @starthour
    ELSE IF @starthour < @endhour
    -- Delete records >= @starthour and <= @endhourur
      DELETE tempdb.dbo.SQLErrorLog
      WHERE PATINDEX(@text,[Text]) > 0 
       AND ([SQLInstance] = @sqlinstance 
            OR @sqlinstance IS NULL)
       AND DATEPART(hh, [LogDate]) >= @starthour
       AND DATEPART(hh, [LogDate]) <= @endhour
    ELSE IF @starthour > @endhour
    -- Delete records >= @starthour or <= @endhour
      DELETE tempdb.dbo.SQLErrorLog
      WHERE PATINDEX(@text,[Text]) > 0 
       AND ([SQLInstance] = @sqlinstance 
            OR @sqlinstance IS NULL)
       AND (DATEPART(hh, [LogDate]) >= @starthour 
            OR DATEPART(hh, [LogDate]) <= @endhour)
    ELSE  -- undefined condition: log error
     BEGIN
       DECLARE @proc VARCHAR(500)
       SELECT @proc = DB_NAME() + '.' + OBJECT_SCHEMA_NAME (@@PROCID) + '.' + OBJECT_NAME(@@PROCID)
       RAISERROR ( 'Proc: %s: Undefined condition found in exclusion table, please contact DBA. Offending line: %s, %s, %u, %u', 10, 1
                   , @proc, @sqlinstance, @text, @starthour ,@endhour) WITH LOG
     END
  FETCH NEXT FROM cleanup INTO @text, @sqlinstance, @starthour, @endhour
  END
CLOSE cleanup
DEALLOCATE cleanup
GO


