DROP XML SCHEMA COLLECTION EmployeeSchema
GO
/*
	A schema may look very complex if there are lots of elements,
	validations and restrictions. Many of the schemas that you might
	use in a production environment would be very long and complex. 

	To simplify the schema development and to make it easier to understand
	and manage, you can define your own custom types. These types can be used
	as children of a "complexType". 

	The following example is using the Schema that we defined in the previous
	session. I have modified the previous schema, to demonstrate the usage of
	a custom type. In this version, I have defined a type named "AgeType". 
	You can find the definition of the type at the bottom of the schema definition. 
	Note that I have defined an element which is of type "AgeType" in the schema 
	definition.

	Use of custom "types" will make the schema development simpler and easier.
	It helps you to re-use the validation rules defined, in multiple instances
	of the type. 
*/
CREATE XML SCHEMA COLLECTION EmployeeSchema AS '
<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
  <xs:element name="Employee">
    <xs:complexType>
      <xs:sequence>
        <xs:element name="FullName" type="xs:string" />
        <xs:element name="Age" type="AgeType" />
        <xs:element name="Nationality">
          <xs:simpleType>
            <xs:restriction base="xs:string">
              <xs:enumeration value="Indian" />
              <xs:enumeration value="British" />
              <xs:enumeration value="American" />
            </xs:restriction>
          </xs:simpleType>
        </xs:element>
        <xs:element name="Gender">
          <xs:simpleType>
            <xs:restriction base="xs:string">
              <xs:pattern value="male|female" />
            </xs:restriction>
          </xs:simpleType>
        </xs:element>
        <xs:element name="Salary">
          <xs:simpleType>
            <xs:restriction base="xs:decimal">
              <xs:fractionDigits value="2" />
            </xs:restriction>
          </xs:simpleType>
        </xs:element>
      </xs:sequence>
      <xs:attribute name="EmployeeNumber" use="required" >
        <xs:simpleType>
          <xs:restriction base="xs:integer">
            <xs:totalDigits value="6" />
          </xs:restriction>
        </xs:simpleType>
      </xs:attribute>
      <xs:attribute name="LoginName" use="required">
        <xs:simpleType>
          <xs:restriction base="xs:string">
            <xs:minLength value="6" />
            <xs:maxLength value="8" />
            <xs:pattern value="([a-z])*" />
          </xs:restriction>
        </xs:simpleType>
      </xs:attribute>
    </xs:complexType>
  </xs:element>
  <xs:simpleType name="AgeType">
    <xs:restriction base="xs:integer">
      <xs:minInclusive value="18" />
      <xs:maxInclusive value="65" />
    </xs:restriction>
  </xs:simpleType>
  </xs:schema>
'
GO

/*
The following is a correct XML value that validates with the above SCHEMA. The
XML value is the same that we used in the previous session. The only change
that we did in this version is a restructuring of the schema, which does not
affect the validation rules. 
*/
DECLARE @emp AS XML(EmployeeSchema)
SET @emp = '
<Employee EmployeeNumber="1001" LoginName="jacobs" >
	<FullName>Jacob</FullName>
	<Age>30</Age>
	<Nationality>Indian</Nationality>
	<Gender>male</Gender>
	<Salary>10000.00</Salary>
</Employee>
'
