/**********************************************
 * StairwayToSQLCLR-04-07-UnsafeTests2.sql
 * 
 * Example Code for
 * Stairway to SQLCLR - Level 4: Security (EXTERNAL_ACCESS and UNSAFE Assemblies)
 * http://www.sqlservercentral.com/articles/Stairway+Series/112888/
 * 
 * Copyright (C) 2014 Solomon Rutzky. All Rights Reserved.
 * 
 **********************************************/

USE [StairwayToSQLCLR];
SET ANSI_NULLS ON;
SET QUOTED_IDENTIFIER ON;
SET NOCOUNT ON;
GO


----------------------------------------------------------

-- The following CREATE ASSEMBLY statement actually succeeds because the only code within 
-- the Assembly that violates any of the restrictions can only be detected at runtime.
CREATE ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck]
	AUTHORIZATION [dbo]
	FROM 'C:\TEMP\StairwayToSQLCLR\Level-04\Assemblies\StairwayToSQLCLR-04-Security2_RuntimeCheck.dll'
	WITH PERMISSION_SET = SAFE;
-- no errors
GO

-- Now that the Assembly (containing code that should not be allowed to run) is loaded,
-- we should be able to create the T-SQL wrapper object that will let us execute the
-- .Net code within the Assembly.
CREATE PROCEDURE [dbo].[StairwayToSQLCLR_04_Beep]
	@NoElBeepo BIT = 0
	AS EXTERNAL	NAME
	[StairwayToSQLCLR-04-Security2_RuntimeCheck].[SqlClrSecurity_RuntimeCheck].[Beep];
GO
-- no errors

----------------------------------------------------------
-- We have loaded an Assembly containing unsafe code and created the T-SQL wrapper
-- object that references that unsafe code. Does this mean that we were able to
-- bypass security? Let's see.

EXEC dbo.StairwayToSQLCLR_04_Beep;
-- error: 
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_04_Beep, Line 0
   System.Security.HostProtectionException: Attempted to perform an operation
   that was forbidden by the CLR host.

   The protected resources (only available with full trust) were: All
   The demanded resources were: UI
*/

-- http://msdn.microsoft.com/en-US/library/ms403285.aspx
-- look for "Beep"


-- Since this is a runtime test, what if the offending code were never executed:
EXEC dbo.StairwayToSQLCLR_04_Beep @NoElBeepo = 1;
-- error: 
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_04_Beep, Line 0
   System.Security.HostProtectionException: Attempted to perform an operation
   that was forbidden by the CLR host.

   The protected resources (only available with full trust) were: All
   The demanded resources were: UI
*/

-- It seems that SQL Server scans the whole method for runtime checks, and not
-- the specific / requested code path.


-- The .Net method with the forbidden function call can't be executed while the
-- Assembly is set to SAFE, but now that the Assembly exists, can it be ALTERed
-- to be UNSAFE?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck]
	WITH PERMISSION_SET = UNSAFE;
-- error:
/*
Msg 10327, Level 14, State 1, Line 1
ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-04-Security2_RuntimeCheck' failed because
	assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' is not authorized for
	PERMISSION_SET = UNSAFE.  The assembly is authorized when either of the following
	is true: the database owner (DBO) has UNSAFE permission and the database has the
	TRUSTWORTHY database property on; or the assembly is signed with a certificate or
	an asymmetric key that has a corresponding login with UNSAFE permission.
*/


-- But we were able to set the other two Assemblies to UNSAFE. Can we at least set
-- this new Assembly to EXTERNAL_ACCESS, just to see if it will work?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck]
	WITH PERMISSION_SET = EXTERNAL_ACCESS;
/*
Msg 10327, Level 14, State 1, Line 1
   ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-04-Security2_RuntimeCheck' failed
   because assembly 'StairwayToSQLCLR-04-Security2_RuntimeCheck' is not authorized
   for PERMISSION_SET = EXTERNAL_ACCESS.  The assembly is authorized when either of
   the following is true: the database owner (DBO) has EXTERNAL ACCESS ASSEMBLY
   permission and the database has the TRUSTWORTHY database property on; or the
   assembly is signed with a certificate or an asymmetric key that has a
   corresponding login with EXTERNAL ACCESS ASSEMBLY permission.
*/

-- Take a look at the Assembly meta-data for clues:
SELECT [name], [clr_name] FROM sys.assemblies WHERE is_user_defined = 1 ORDER BY [name] ASC;
-- Pay attention to the "publickeytoken=" portion of the [clr_name] field.
-- The "publickeytoken" value is the same for the CreateAssemblyCheck and
-- ExternalAccess Assemblies. But the value for the RuntimeCheck Assembly
-- is different.


-- The Login, MrStairwayToSQLCLR, is linked to the Asymmetric Key that was created
-- from the Certificate used to sign the ExternalAccess and CreateAssemblyCheck
-- Assemblies. But the Assembly, StairwayToSQLCLR-04-Security2_RuntimeCheck,
-- was signed with a different Certificate. In order for this new Assembly to be
-- able to be set to either EXTERNAL_ACCESS or UNSAFE, we need to create another
-- Asymmetric Key and then another Login based on that Key.


-- Step 1: Create an Asymmetric Key based on info stored in the Assembly's source DLL file.
-- Do this within an EXEC to keep the database context as StairwayToSQLCLR
EXEC ('USE [master];
		CREATE ASYMMETRIC KEY [StairwayToSQLCLR-04-Key2]
			AUTHORIZATION [dbo]
			FROM EXECUTABLE FILE = ''C:\TEMP\StairwayToSQLCLR\Level-04\Assemblies\StairwayToSQLCLR-04-Security2_RuntimeCheck.dll'';
');
-- no errors


-- Step 2: Create a Login based on the Asymmetric Key we just created in Step 1.
EXEC ('USE [master];
		CREATE LOGIN [MrsStairwayToSQLCLR]
			FROM ASYMMETRIC KEY [StairwayToSQLCLR-04-Key2];
');
-- no errors


-- Step 3: Grant the Login we just created in Step 2 the desired permission.
EXEC ('USE [master];
		GRANT UNSAFE ASSEMBLY TO [MrsStairwayToSQLCLR];
');
-- no errors


-- We only granted UNSAFE ASSEMBLY permission, but can we also set the
-- Assembly to EXTERNAL_ACCESS?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck]
	WITH PERMISSION_SET = EXTERNAL_ACCESS;
-- no errors due to the "UNSAFE ASSEMBLY" permission implying the 
-- "EXTERNAL ACCESS ASSEMBLY" permission.


EXEC dbo.StairwayToSQLCLR_04_Beep;
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_04_Beep, Line 0
   System.Security.HostProtectionException: Attempted to perform an operation
   that was forbidden by the CLR host.

   The protected resources (only available with full trust) were: All
   The demanded resources were: UI
*/


ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck]
	WITH PERMISSION_SET = UNSAFE;
-- no errors


EXEC dbo.StairwayToSQLCLR_04_Beep;
-- no errors

GO

----------------------------------------------------------
-- Clean Up!
/*
IF (OBJECT_ID(N'dbo.StairwayToSQLCLR_04_Beep') IS NOT NULL)
BEGIN
	PRINT 'Dropping procedure [StairwayToSQLCLR_04_Beep] ...';
	DROP PROCEDURE [dbo].[StairwayToSQLCLR_04_Beep];
	PRINT 'Done.';
	PRINT '';
END;

IF (ASSEMBLYPROPERTY(N'StairwayToSQLCLR-04-Security2_RuntimeCheck', 'SimpleName') IS NOT NULL)
BEGIN
	PRINT 'Dropping Assembly [StairwayToSQLCLR-04-Security2_RuntimeCheck] ...';
	DROP ASSEMBLY [StairwayToSQLCLR-04-Security2_RuntimeCheck];
	PRINT 'Done.';
	PRINT '';
END;
*/
