/**********************************************
 * StairwayToSQLCLR-04-05-TrustworthyVsAsymmetricKeyTests.sql
 * 
 * Example Code for
 * Stairway to SQLCLR - Level 4: Security (EXTERNAL_ACCESS and UNSAFE Assemblies)
 * http://www.sqlservercentral.com/articles/Stairway+Series/112888/
 * 
 * Copyright (C) 2014 Solomon Rutzky. All Rights Reserved.
 * 
 **********************************************/

USE [StairwayToSQLCLR];
SET ANSI_NULLS ON;
SET QUOTED_IDENTIFIER ON;
SET NOCOUNT ON;
GO


----------------------------------------------------------
-- In the previous script, we set [StairwayToSQLCLR-04-Security2_ExternalAccess]
-- to EXTERNAL_ACCESS.  Can we set it to UNSAFE as well?

ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = UNSAFE;
-- no error.

-- Yes, because we had changed the DB-level setting of TRUSTWORTHY to ON.


----------------------------------------------------------
-- In "Stairway To SQLCLR: Level 3" we tried to load the following Assembly
-- but couldn't.  SAFE did not work due to the code needing the UNSAFE
-- permission. Loading as either EXTERNAL_ACCESS or UNSAFE failed due to
-- no ability to load any Assemblies with anything but SAFE permissions.
-- This time, however, we have set TRUSTWORTHY ON.

CREATE ASSEMBLY [StairwayToSQLCLR-04-Security2_CreateAssemblyCheck]
	AUTHORIZATION [dbo]
	FROM 'C:\TEMP\StairwayToSQLCLR\Level-04\Assemblies\StairwayToSQLCLR-04-Security2_CreateAssemblyCheck.dll'
	WITH PERMISSION_SET = UNSAFE;
-- no errors


-- When TRUSTWORTHY is to set ON, _any_ Assembly can be set to either EXTERNAL_ACCESS or UNSAFE.
-- There are no restrictions, which is why setting TRUSTWORTHY ON is the less preferred method.

-------------------------------
-- Clean up prior tests so that we can attempt the preferred method of allowing Assemblies to
-- be either EXTERNAL_ACCESS or UNSAFE.

DROP ASSEMBLY [StairwayToSQLCLR-04-Security2_CreateAssemblyCheck];
GO
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = SAFE;
GO
ALTER DATABASE [StairwayToSQLCLR] SET TRUSTWORTHY OFF;
GO


-------------------------------

-- Step 1: Create an Asymmetric Key based on info stored in the Assembly's source DLL file.
-- Do this within an EXEC to keep the database context as StairwayToSQLCLR
EXEC ('USE [master];
		CREATE ASYMMETRIC KEY [StairwayToSQLCLR-04-Key1]
			AUTHORIZATION [dbo]
			FROM EXECUTABLE FILE = ''C:\TEMP\StairwayToSQLCLR\Level-04\Assemblies\StairwayToSQLCLR-04-Security2_ExternalAccess.dll'';
');
-- no errors


-- Did it work?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = EXTERNAL_ACCESS;
/*
Msg 10327, Level 14, State 1, Line 1
ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-04-Security2_ExternalAccess' failed
   because assembly 'StairwayToSQLCLR-04-Security2_ExternalAccess' is not authorized
   for PERMISSION_SET = EXTERNAL_ACCESS.  The assembly is authorized when either of
   the following is true: the database owner (DBO) has EXTERNAL ACCESS ASSEMBLY
   permission and the database has the TRUSTWORTHY database property on; or the
   assembly is signed with a certificate or an asymmetric key that has a
   corresponding login with EXTERNAL ACCESS ASSEMBLY permission.
*/


-- Step 2: Create a Login based on the Asymmetric Key we just created in Step 1.
CREATE LOGIN [MrStairwayToSQLCLR]
	FROM ASYMMETRIC KEY [StairwayToSQLCLR-04-Key1];
-- no errors


-- Did it work?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = EXTERNAL_ACCESS;
-- same error as in Step 1 above.


-- Step 3: Grant the Login we just created in Step 2 the desired permission.
EXEC ('USE [master];
		GRANT EXTERNAL ACCESS ASSEMBLY TO [MrStairwayToSQLCLR];
');
-- no errors


-- NOW did it work?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = EXTERNAL_ACCESS;
-- no errors !!

-- Test it fully.
-- External / Regular connection with Impersonation
EXEC dbo.StairwayToSQLCLR_04_WhoAmI @ServerName = '(local)', @Impersonate = 1;
-- no errors !!


-------------------------------

-- Can we set it to UNSAFE like we did earlier (first test in this script)?
ALTER ASSEMBLY [StairwayToSQLCLR-04-Security2_ExternalAccess] WITH PERMISSION_SET = UNSAFE;
/*
Msg 10327, Level 14, State 1, Line 1
ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-04-Security2_ExternalAccess' failed
   because assembly 'StairwayToSQLCLR-04-Security2_ExternalAccess' is not authorized
   for PERMISSION_SET = UNSAFE.  The assembly is authorized when either of the
   following is true: the database owner (DBO) has UNSAFE ASSEMBLY permission and
   the database has the TRUSTWORTHY database property on; or the assembly is signed
   with a certificate or an asymmetric key that has a corresponding login with
   UNSAFE ASSEMBLY permission.
*/


-- Hence one benefit of using a Login based on an Asymmetric Key or Certificate
-- (instead of setting TRUSTWORTHY to ON) is that an Assembly can be allowed to
-- be set to EXTERNAL_ACCESS _without_ being allowed to be set to UNSAFE!
