/**********************************************
 * StairwayToSQLCLR-03-06-InternalDataAccessTest.sql
 * 
 * Example Code for
 * Stairway to SQLCLR - Level 3: Security (General and SAFE Assemblies)
 * http://www.sqlservercentral.com/articles/Stairway+Series/109905/
 * 
 * Copyright (C) 2014 Solomon Rutzky. All Rights Reserved.
 * 
 **********************************************/

USE [StairwayToSQLCLR];
SET ANSI_NULLS ON;
SET QUOTED_IDENTIFIER ON;
SET NOCOUNT ON;
GO


----------------------------------------------------------
-- Before we begin, create an indicator of the current session that we can test
-- for. Highlight from the DECLARE down to (and including) the 2nd SELECT and run.
DECLARE @GUID UNIQUEIDENTIFIER;
SET @GUID = NEWID();

SELECT	@GUID AS [@GUID],
		CONVERT(VARBINARY(50), @GUID) AS [@GUID -> VARBINARY];

SET CONTEXT_INFO @GUID;

-- run the following statement again, by itself, and the values won't change.
SELECT	CONTEXT_INFO() AS [CONTEXT_INFO()],
		CONVERT(UNIQUEIDENTIFIER, CONTEXT_INFO()) AS [CONTEXT_INFO() -> UNIQUEIDENTIFIER];


-- run the following to see that the CONTEXT_INFO value truly is session-specific
SELECT [context_info], *
FROM sys.dm_exec_sessions
ORDER BY 1 DESC;


-- In the following two tests, pay close attention to the values returned
-- in the following fields: @@SPID, CONTEXT_INFO(), and SESSION_USER.


-- First do a straight T-SQL call directly to the function as a control
-- to make sure that the function works and see what values are returned
-- under "normal" conditions; this should not error

SELECT * FROM dbo.WhoAmI();
-- [@@SPID], [CONTEXT_INFO()], and SESSION_USER fields will have the expected values.

-------------------------------------

-- Next run the SQLCLR Stored Procedure that does a simple
-- "SELECT * FROM dbo.WhoAmI()", and have the Proc use the internal
-- data access (i.e. Context Connection) which is the current process /
-- session (i.e. @@SPID). All fields should return the same values
-- that they did in the prior test.

EXEC dbo.StairwayToSQLCLR_03_WhoAmI; -- default for @ServerName = NULL
-- [@@SPID], [CONTEXT_INFO()], and SESSION_USER fields will have the expected values.


----------------------------------------------------------

-- Now run the SQLCLR Stored Procedure again, but this time use
-- a regular / external connection.

EXEC dbo.StairwayToSQLCLR_03_WhoAmI @ServerName = '(local)';
-- error:
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_03_WhoAmI, Line 0
System.Security.SecurityException
*/


-- SAFE mode will not allow the process to go outside of SQL Server, even
-- if only to come right back into the same instance. So, change the
-- Assembly's PERMISSION_SET so that it is not restricted.

ALTER ASSEMBLY [StairwayToSQLCLR-03-Security] WITH PERMISSION_SET = EXTERNAL_ACCESS;
/*
Msg 10327, Level 14, State 1, Line 1
ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-03-Security' failed because assembly
	'StairwayToSQLCLR-03-Security' is not authorized for PERMISSION_SET =
	EXTERNAL_ACCESS.  The assembly is authorized when either of the following is
	true: the database owner (DBO) has EXTERNAL ACCESS ASSEMBLY permission and the
	database has the TRUSTWORTHY database property on; or the assembly is signed
	with a certificate or an asymmetric key that has a corresponding login with
	EXTERNAL ACCESS ASSEMBLY permission.
*/

-- Hence, having permission to run ALTER ASSEMBLY does not mean that the command
-- will succeed.

----------------------------------------------------------

-- Impersonation will be covered in Level 4, but just to see what
-- happens when the Assembly is set to SAFE, run the following:

EXEC dbo.StairwayToSQLCLR_03_WhoAmI @ServerName = NULL, @Impersonate = 1;
-- error:
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_03_WhoAmI, Line 0
System.Security.SecurityException
*/

-- The Context Connection (i.e. internal data access) is not available when
-- using impersonation, but that is not the reason for the error. The error
-- is caused by the .Net code attempting to get the WindowsIdentity property
-- of the SqlContext object. This information is needed in order to do the
-- impersonation, but it is off limits for Assemblies marked as SAFE:
---- http://technet.microsoft.com/en-us/library/ms131108.aspx
---- Refer to "Retrieving Windows Identity" section:
----   "Only assemblies marked with EXTERNAL_ACCESS or UNSAFE permissions can
----    access this property."

----------------------------------------------------------

