/**********************************************
 * StairwayToSQLCLR-03-04-RuntimeCheckTest.sql
 * 
 * Example Code for
 * Stairway to SQLCLR - Level 3: Security (General and SAFE Assemblies)
 * http://www.sqlservercentral.com/articles/Stairway+Series/109905/
 * 
 * Copyright (C) 2014 Solomon Rutzky. All Rights Reserved.
 * 
 **********************************************/

USE [StairwayToSQLCLR];
SET ANSI_NULLS ON;
SET QUOTED_IDENTIFIER ON;
SET NOCOUNT ON;
GO

-- Each query below is intended to run individually by highlighting
-- one query at a time.

----------------------------------------------------------
-- The following two CREATE ASSEMBLY statements will fail without even
-- checking the code inside the Assembly due to the database not allowing
-- any Assemblies to be set to UNSAFE or even EXTERNAL_ACCESS. This is
-- because neither of the two conditions required for allowing Assemblies
-- to be set to UNSAFE or EXTERNAL_ACCESS have been met, and that is
-- good as we (currently) only want to allow SAFE Assemblies.


CREATE ASSEMBLY [StairwayToSQLCLR-03-Security_RuntimeCheck]
	AUTHORIZATION [dbo]
	FROM 'C:\TEMP\StairwayToSQLCLR\Level-03\Assemblies\StairwayToSQLCLR-03-Security_RuntimeCheck.dll'
	WITH PERMISSION_SET = UNSAFE;
/*
Msg 10327, Level 14, State 1, Line 1
CREATE ASSEMBLY for assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' failed because
   assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' is not authorized for
   PERMISSION_SET = UNSAFE.  The assembly is authorized when either of the following is true:
   the database owner (DBO) has UNSAFE ASSEMBLY permission and the database has the TRUSTWORTHY
   database property on; or the assembly is signed with a certificate or an asymmetric key that
   has a corresponding login with UNSAFE ASSEMBLY permission.
*/


CREATE ASSEMBLY [StairwayToSQLCLR-03-Security_RuntimeCheck]
	AUTHORIZATION [dbo]
	FROM 'C:\TEMP\StairwayToSQLCLR\Level-03\Assemblies\StairwayToSQLCLR-03-Security_RuntimeCheck.dll'
	WITH PERMISSION_SET = EXTERNAL_ACCESS;
/*
Msg 10327, Level 14, State 1, Line 1
CREATE ASSEMBLY for assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' failed because
	assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' is not authorized for
	PERMISSION_SET = EXTERNAL_ACCESS.  The assembly is authorized when either of the following
	is true: the database owner (DBO) has EXTERNAL ACCESS ASSEMBLY permission and the database
	has the TRUSTWORTHY database property on; or the assembly is signed with a certificate or
	an asymmetric key that has a corresponding login with EXTERNAL ACCESS ASSEMBLY permission.
*/

-------------------------------
-- The following CREATE ASSEMBLY statement actually succeeds because the only code within 
-- the Assembly that violates any of the restrictions can only be detected at runtime.


CREATE ASSEMBLY [StairwayToSQLCLR-03-Security_RuntimeCheck]
	AUTHORIZATION [dbo]
	FROM 'C:\TEMP\StairwayToSQLCLR\Level-03\Assemblies\StairwayToSQLCLR-03-Security_RuntimeCheck.dll'
	WITH PERMISSION_SET = SAFE;
-- no errors
GO

-- Now that the Assembly (containing code that should not be allowed to run) is loaded,
-- we should be able to create the T-SQL wrapper object that will let us execute the
-- .Net code within the Assembly.

CREATE PROCEDURE [dbo].[StairwayToSQLCLR_03_Beep]
	@NoElBeepo BIT = 0
	AS EXTERNAL NAME
	[StairwayToSQLCLR-03-Security_RuntimeCheck].[SqlClrSecurityRuntime].[Beep];
GO
-- no errors

----------------------------------------------------------
-- We have loaded an Assembly containing unsafe code and created the T-SQL wrapper
-- object that references that unsafe code. Does this mean that we were able to
-- bypass security? Let's see.

EXEC dbo.StairwayToSQLCLR_03_Beep;
-- error: 
/*
Msg 6522, Level 16, State 1, Procedure StairwayToSQLCLR_03_Beep, Line 0
System.Security.HostProtectionException: Attempted to perform an operation that was forbidden
	by the CLR host.

The protected resources (only available with full trust) were: All
The demanded resources were: UI
*/

-- http://msdn.microsoft.com/en-US/library/ms403285.aspx
-- look for "Beep"


-- What if we try to execute the .Net method with the forbidden code, but don't
-- execute the forbidden code itself due to it being in an IF block that will
-- be skipped if we pass in a parameter set to true?

EXEC dbo.StairwayToSQLCLR_03_Beep @NoElBeepo = 1;
-- same error as above, even though we attempted to exit without calling the Beep() function.


-- The .Net method with the forbidden function call can't be executed while the
-- Assembly is set to SAFE, but now that the Assembly exists, can it be ALTERed
-- to be UNSAFE?

ALTER ASSEMBLY [StairwayToSQLCLR-03-Security_RuntimeCheck]
	WITH PERMISSION_SET = UNSAFE;
-- error:
/*
Msg 10327, Level 14, State 1, Line 1
ALTER ASSEMBLY for assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' failed because
	assembly 'StairwayToSQLCLR-03-Security_RuntimeCheck' is not authorized for
	PERMISSION_SET = UNSAFE.  The assembly is authorized when either of the following
	is true: the database owner (DBO) has UNSAFE permission and the database has the
	TRUSTWORTHY database property on; or the assembly is signed with a certificate or
	an asymmetric key that has a corresponding login with UNSAFE permission.
*/


GO

----------------------------------------------------------
-- Clean Up!
/*
IF (OBJECT_ID(N'dbo.StairwayToSQLCLR_03_Beep') IS NOT NULL)
BEGIN
	PRINT 'Dropping procedure [StairwayToSQLCLR_03_Beep] ...';
	DROP PROCEDURE [dbo].[StairwayToSQLCLR_03_Beep];
	PRINT 'Done.';
	PRINT '';
END;

IF (ASSEMBLYPROPERTY(N'StairwayToSQLCLR-03-Security_RuntimeCheck', 'SimpleName') IS NOT NULL)
BEGIN
	PRINT 'Dropping Assembly [StairwayToSQLCLR-03-Security_RuntimeCheck] ...';
	DROP ASSEMBLY [StairwayToSQLCLR-03-Security_RuntimeCheck];
	PRINT 'Done.';
	PRINT '';
END;
*/
