/**********************************************
 * Example Code for
 * Stairway to SQLCLR - Level 3: Security (General and SAFE Assemblies)
 * http://www.sqlservercentral.com/articles/Stairway+Series/109905/
 * 
 * Copyright (C) 2014 Solomon Rutzky. All Rights Reserved.
 * 
 **********************************************/

using System; // contains types (String / Int64 / Guid), Exception, and ArgumentException
using System.Data; // contains CommandType, ConnectionState, SqlDbType, and ParameterDirection enums
using System.Data.SqlClient; // contains SqlConnection, SqlCommand, SqlParameter, and SqlDataReader
using System.Data.SqlTypes; // contains the Sql* types for input / output params
using System.Security.Principal; // contains WindowsImpersonationContext
using Microsoft.SqlServer.Server; // contains SqlFacet(Attribute), SqlContext, (System)DataAccessKind

public class SqlClrSecurity
{
	[Microsoft.SqlServer.Server.SqlProcedure(Name = "StairwayToSQLCLR_03_WhoAmI")]
	public static void WhoAmI([SqlFacet(MaxSize = 200)] SqlString ServerName,
		SqlByte Impersonate, SqlString Message)
	{
		string connectionString = "context connection = true";
		WindowsImpersonationContext impersonationIdentity = null;

		if (!ServerName.IsNull && ServerName.Value.Trim() != string.Empty)
		{
			connectionString =
				String.Concat(	"server=",
								ServerName.Value,
								";trusted_connection=true;");
		}

		string query = @"SELECT	* FROM StairwayToSQLCLR.dbo.WhoAmI();";

		SqlDataReader dataReader = null;
		SqlConnection sqlConnection = new SqlConnection(connectionString);
		SqlCommand sqlCommand = new SqlCommand(query, sqlConnection);
		sqlCommand.CommandType = CommandType.Text;

		try
		{
			// if current user is Windows Auth (i.e. NOT DB Auth) then
			// impersonate for remote calls else remote connections
			// will be made by SQL Server service account
			if (!Impersonate.IsNull &&
				  (
					(Impersonate.Value == 1 && SqlContext.WindowsIdentity != null)
				   || Impersonate.Value == 2
				  )
				)
			{
				impersonationIdentity = SqlContext.WindowsIdentity.Impersonate();
			}

			sqlConnection.Open();

			if (!Message.IsNull && Message.Value != "")
			{
				SqlContext.Pipe.Send(Message.Value); // PRINT @Message
			}

			if (impersonationIdentity != null)
			{
				impersonationIdentity.Undo();
			}

			dataReader = sqlCommand.ExecuteReader();

			SqlContext.Pipe.Send(dataReader);
		}
		catch (System.Security.SecurityException ex)
		{
			// If this is a Permissions error,
			// give the user some useful info to help correct it
			if (ex.PermissionType.FullName ==
						"System.Data.SqlClient.SqlClientPermission"
				|| ex.PermissionType.FullName ==
						"System.Security.Permissions.SecurityPermission")
			{
				throw new System.Security.SecurityException(
					   "\n\nSecurity exception.\n\n"
					+ "Please run the following:\n\n"
					+ "\tALTER ASSEMBLY [StairwayToSQLCLR-03-Security] "
					+ "WITH PERMISSION_SET = EXTERNAL_ACCESS;\n"
					+ "\n\n");
			}

			throw;
		}
		catch (Exception ex)
		{
			if (impersonationIdentity != null)
			{
				impersonationIdentity.Undo();
			}
			throw;
		}
		finally
		{
			// make sure to clean up external resources!
			if (dataReader != null && !dataReader.IsClosed)
			{
				dataReader.Close();
			}

			if (sqlConnection.State != ConnectionState.Closed)
			{
				sqlConnection.Close();
			}

			if (impersonationIdentity != null)
			{
				impersonationIdentity.Undo();
			}
		}


		return;
	}

	[Microsoft.SqlServer.Server.SqlProcedure(
		Name = "StairwayToSQLCLR_03_InsertRowCLR")]
	public static void InsertRowCLR(SqlString Message)
	{
		if (Message.IsNull)
		{
			SqlContext.Pipe.Send("Nothing to INSERT.");
			return;
		}

		string query = @"INSERT INTO StairwayToSQLCLR.dbo.StairwayToSQLCLR_03
			(MessageText) VALUES (@Message);";

		SqlConnection sqlConnection =
			new SqlConnection("context connection = true");
		SqlCommand sqlCommand = new SqlCommand(query, sqlConnection);
		sqlCommand.CommandType = CommandType.Text;

		SqlParameter sqlParamMessage =
			new SqlParameter("@Message", SqlDbType.NVarChar, 4000);
		sqlParamMessage.Direction = ParameterDirection.Input;
		sqlParamMessage.Value = Message.Value;
		sqlCommand.Parameters.Add(sqlParamMessage);

		try
		{
			sqlConnection.Open();

			sqlCommand.ExecuteNonQuery();
		}
		catch (Exception ex)
		{
			throw;
		}
		finally
		{
			if (sqlConnection.State != ConnectionState.Closed)
			{
				sqlConnection.Close();
			}
		}


		return;
	}

}
