﻿using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.SqlServer.Dts.Pipeline;
using Microsoft.SqlServer.Dts.Pipeline.Wrapper;
using Microsoft.SqlServer.Dts.Runtime;
using Microsoft.SqlServer.Dts.Runtime.Wrapper;
using Microsoft.SqlServer;
using System.Xml;


/* Jack D. Corbett
 * corbett.jack@gmail.com
 * 
 * Created to log errors in a dataflow to an Destination.  I use a table.
 * Takes the data columns are mapped to xml file in name-value pair format in the ErrorDetails output.
 * The other columns as follows are passed through as output columns:
 * 
 * ErrorCode - the error code passed to the transformation
 * ErrorDesc - the error description retrieved using the ErrorCode
 * ErrorTask - the Data Flow that has the error retrieved from the TaskName variable
 * ErrorStep - the name of the Error Details component
 * PackageTime - the time the package was started retrieved from the StartTime variable
 * PackageName - The name of the package that called the task
 * UserName -   The user who invoked the package to run.
 * Notes:
 *      I'm not out for anything with this so if you need it, use it, if you extend it, share it.
 *      
 * History
 * ----------
 * 08-18-2008 Created
 * 01-10-2009 (Zach Mattson)Upgraded to BIDS 2008 interfaces, added User and Package collection info.
 */ 
namespace SSIS.Logging
{
    [DtsPipelineComponent(DisplayName = "Error Details", ComponentType = ComponentType.Transform)]
    public class ErrorDetails : PipelineComponent
    {
        private ColumnInfo[] _inputColumnInfos;
        private ColumnInfo[] _outputColumnInfos;

        private string _rowElementName = "fields";
        private string _columnElementName = "field";
        private string _nameAttributeName = "name";
        private string _valueAttributeName = "value";
        private string _strErrorDesc = string.Empty;

        public struct ColumnInfo
            {
               public int bufferColumnIndex;
               public DTSRowDisposition columnDisposition;
               public int lineageID;
               public DataType DataType;
               public string Name;
            }// end struct ColumnInfo
            
        public override void ProvideComponentProperties()
        {
            base.ProvideComponentProperties();

            base.RemoveAllInputsOutputsAndCustomProperties();

            IDTSInput100 input = this.ComponentMetaData.InputCollection.New();
            input.Name = "EDInput";

            IDTSOutput100 output = this.ComponentMetaData.OutputCollection.New();
            output.Name = "EDOutput";

            output.SynchronousInputID = input.ID;

            AddOutputColumns();
           
        } // end sub PovideComponent Properties

        private void AddOutputColumns()
        {
            // xml column
            IDTSOutputColumn100 outColumn = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outColumn.Name = "ErrorDetails";
            outColumn.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 4000, 0, 0, 0);

            // description of error from errorcode
            IDTSOutputColumn100 outErrorDesc = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outErrorDesc.Name = "ErrorDesc";
            outErrorDesc.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 200, 0, 0, 0);
            
            // step of the flow that caused the error - will be this components name
            IDTSOutputColumn100 outErrorStep = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outErrorStep.Name = "ErrorStep";
            outErrorStep.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 200, 0, 0, 0);

            // Data Flow that raised the error
            IDTSOutputColumn100 outErrorTask = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outErrorTask.Name = "ErrorTask";
            outErrorTask.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 200, 0, 0, 0);

            // time of the package
            IDTSOutputColumn100 outPackageTime = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outPackageTime.Name = "PackageTime";
            outPackageTime.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_DATE, 0, 0, 0, 0);
            
            // Package that raised the error
            IDTSOutputColumn100 outPackageName = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outPackageName.Name = "PackageName";
            outPackageName.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 200, 0, 0, 0);

            // User that fired the package
            IDTSOutputColumn100 outUserName = ComponentMetaData.OutputCollection[0].OutputColumnCollection.New();
            outUserName.Name = "UserName";
            outUserName.SetDataTypeProperties(Microsoft.SqlServer.Dts.Runtime.Wrapper.DataType.DT_WSTR, 128,0, 0, 0);
        }   

        public override DTSValidationStatus Validate()
        {
            DTSValidationStatus status;

            status = base.Validate();

            if (status == DTSValidationStatus.VS_ISVALID)
            {
                if ((this.ComponentMetaData.InputCollection.Count != 1) || (this.ComponentMetaData.OutputCollection.Count != 1))
                {
                    bool cancel = false;

                    this.ComponentMetaData.FireError(0, ComponentMetaData.Name, "Cannot have more than 1 input or output", string.Empty, 0, out cancel);

                    status = DTSValidationStatus.VS_ISCORRUPT;
                }
            }

            return status;
        } // end function Validate

        public override void PreExecute()
        {
            base.PreExecute();

            IDTSInput100 input = this.ComponentMetaData.InputCollection[0];
            
            _inputColumnInfos = new ColumnInfo[input.InputColumnCollection.Count];
            for (int x = 0; x < input.InputColumnCollection.Count; x++)
            {
                IDTSInputColumn100 column = input.InputColumnCollection[x];
                _inputColumnInfos[x] = new ColumnInfo();
                _inputColumnInfos[x].bufferColumnIndex = BufferManager.FindColumnByLineageID(input.Buffer, column.LineageID);
                _inputColumnInfos[x].columnDisposition = column.ErrorRowDisposition;
                _inputColumnInfos[x].lineageID = column.LineageID;
                _inputColumnInfos[x].DataType = column.DataType;
                _inputColumnInfos[x].Name = column.Name;
            }
            
            IDTSOutput100 output = this.ComponentMetaData.OutputCollection[0];
            _outputColumnInfos = new ColumnInfo[output.OutputColumnCollection.Count];

            for (int i = 0; i < output.OutputColumnCollection.Count; i++)
            {
                IDTSOutputColumn100 column = output.OutputColumnCollection[i];
                _outputColumnInfos[i] = new ColumnInfo();
                _outputColumnInfos[i].bufferColumnIndex = BufferManager.FindColumnByLineageID(input.Buffer, column.LineageID);
                _outputColumnInfos[i].columnDisposition = column.ErrorRowDisposition;
                _outputColumnInfos[i].lineageID = column.LineageID;
                _outputColumnInfos[i].Name = column.Name;
            }
        
        }// end sub PreExecute

        public override void ProcessInput(int inputID, PipelineBuffer buffer)
        {
            base.ProcessInput(inputID, buffer);
            int defaultOutputId = -1;

            if (buffer.EndOfRowset == false)
            {
                try
                {
                    while (buffer.NextRow())
                    {
                        /// If the columnInfos array has zero dimensions, then 
                        /// no input columns have been selected for the component. 
                        /// Direct the row to the default output.
                        if (_inputColumnInfos.Length == 0)
                            buffer.DirectRow(defaultOutputId);

                        StringBuilder sb = new StringBuilder();
                        using (XmlWriter writer = XmlWriter.Create(sb))
                        {
                            writer.WriteStartElement(_rowElementName);

                            /// Iterate the columns in the columnInfos array.
                            for (int i = 0; i < _inputColumnInfos.Length; i++)
                            {
                                ColumnInfo colInfo = _inputColumnInfos[i];

                                // using Contains because it could be named Component - ErrorCode/ErrorColumn
                                if (!(colInfo.Name.Contains("ErrorCode")) && !(colInfo.Name.Contains("ErrorColumn")))
                                {
                                    writer.WriteStartElement(_columnElementName);
                                    writer.WriteAttributeString(_nameAttributeName, colInfo.Name);
                                    
                                    
                                    /// Is the column null?
                                    if (!buffer.IsNull(colInfo.bufferColumnIndex))
                                    {
                                        /// No, process it.
                                        string columnValue = buffer[colInfo.bufferColumnIndex].ToString();

                                        writer.WriteAttributeString(_valueAttributeName,columnValue);
                                    }
                                    else
                                    {
                                        writer.WriteAttributeString(_valueAttributeName, "true");
                                    }

                                    writer.WriteEndElement();
                                }// end if ErrorCode or ErrorColumn
                                else // if errorcode then get error description
                                {
                                    if (colInfo.Name.Contains("ErrorCode"))
                                    {
                                        _strErrorDesc = this.ComponentMetaData.GetErrorDescription(buffer.GetInt32(colInfo.bufferColumnIndex));
                                    }
                                }

                            }// end for loop over _inputColumnInfos

                            writer.WriteEndElement();
                        }// end using XMLWriter
                        
                        IDTSVariables100 objVars;

                        // loop through output columns to assign data
                        for (int i = 0; i < _outputColumnInfos.Length; i++)
                        {
                            ColumnInfo colInfo = _outputColumnInfos[i];
                            
                            switch (colInfo.Name.ToUpper())
                            {
                                case "ERRORDESC":
                                    buffer.SetString(colInfo.bufferColumnIndex, _strErrorDesc);
                                    break;
                                case "ERRORDETAILS":
                                    buffer.SetString(colInfo.bufferColumnIndex, sb.ToString());
                                    break;
                                case "ERRORSTEP":
                                    buffer.SetString(colInfo.bufferColumnIndex, this.ComponentMetaData.Name);
                                    break;
                                case "ERRORTASK":
                                    this.VariableDispenser.LockForRead("System::TaskName");
                                    this.VariableDispenser.GetVariables(out objVars);
                                    buffer.SetString(colInfo.bufferColumnIndex, objVars[0].Value.ToString());
                                    objVars.Unlock();
                                    break;
                                case "PACKAGENAME":
                                    this.VariableDispenser.LockForRead("System::PackageName");
                                    this.VariableDispenser.GetVariables(out objVars);
                                    buffer.SetString(colInfo.bufferColumnIndex, objVars[0].Value.ToString());
                                    objVars.Unlock();
                                    break;
                                case "USERNAME":
                                    this.VariableDispenser.LockForRead("System::UserName");
                                    this.VariableDispenser.GetVariables(out objVars);
                                    buffer.SetString(colInfo.bufferColumnIndex, objVars[0].Value.ToString());
                                    objVars.Unlock();
                                    break;
                                case "PACKAGETIME":
                                    this.VariableDispenser.LockForRead("System::StartTime");
                                    this.VariableDispenser.GetVariables(out objVars);
                                    buffer.SetDateTime(colInfo.bufferColumnIndex, (DateTime)objVars[0].Value);
                                    objVars.Unlock();
                                    break;
                            } // End switch on colInfo.Name
                        } //end for loop through output columns
                    }//end while
                }//end try
                catch (System.Exception ex)
                {
                    bool Cancel = false;

                    ComponentMetaData.FireError(0, ComponentMetaData.Name, ex.Message, string.Empty, 0, out Cancel);

                    throw new Exception("Could not process input buffer.", ex);
                }// end try catch
            }// end if buffer.EndOfRowset == false
        } // end ProcessInput

        public override void DeleteInput(int inputID)
        {
            throw new Exception("Can't delete input " + inputID.ToString(System.Globalization.CultureInfo.InvariantCulture));
        }

        public override void DeleteOutput(int outputID)
        {
            throw new Exception("Can't delete output " + outputID.ToString(System.Globalization.CultureInfo.InvariantCulture));
        }

        public override void DeleteOutputColumn(int outputID, int outputColumnID)
        {
            throw new Exception("Can't delete output column " + outputColumnID.ToString(System.Globalization.CultureInfo.InvariantCulture) + " from output" + outputID.ToString(System.Globalization.CultureInfo.InvariantCulture));
        }

        public override IDTSOutput100 InsertOutput(DTSInsertPlacement insertPlacement, int outputID)
        {
            throw new Exception("This compononent does not allow output creation.");
        }
        
        public override IDTSInput100 InsertInput(DTSInsertPlacement insertPlacement, int inputID)
        {
            throw new Exception("This compononent does not allow input creation.");
        }

        public override IDTSOutputColumn100 InsertOutputColumnAt(int outputID, int outputColumnIndex, string name, string description)
        {
            throw new Exception("This compononent does not allow output column creation.");
        }
    }//end class ErrorDetails
}
