<#
  .SYNOPSIS
   Script to record database device details on a central monitoring server.

  .DESCRIPTION
   This script can be used to monitor both the growth of physical database devices 
   and the extent to which data and log devices are used (% full).
   
   If used with a file that contains the names of your SQL servers, it enables you to
   monitor space usage on multiple servers from a single source.
   Alternatively, you can use it without such a file to collect device usage of a single 
   SQL server and store the results locally.
   Device size history is stored in the table 'dbo.dbspace'. 

   The idea of feeding the output of DBCC SHOWFILESTATS and DBCC sqlperf(logspace) into a table
   is based on a script by Jonathan Kehayias I once found on MSDN.
   
   SETUP INSTRUCTIONS

   On the server you use to run this script (called Monitor Server below):
    -- Create a SQL database (25-50 MB). 
       Name it 'MonitorDB' or any other name you specify for $ManagementDB in the
       User Defined Parameters below. Make the Windows account running the script dbo.
    -- Copy this script to a directory on the Monitor Server. For the purpose of this example the 
       directory name 'D:\Monitoring' is used.
    -- If you want to monitor device changes on multiple SQL servers:
       -- In this directory, create a file (e.g. 'servers.txt'). This file should contain the SQL instance 
          name(s) whose device usage you want to monitor, one per line, unquoted.
          (For a named instance, use a backslash between server and instance name: server01\sql01).
    -- If you want to only monitor device changes on the local server:
       -- Comment out the line that starts with '$ServerFile =' below by preceding it with a pound sign: '#'
    -- Update the 'User Defined Parameters' in the section below to reflect your setup.
    -- Make sure you have PowerShell version 2 or higher installed on the Monitor Server.
    -- Install the SQLPS module in Powershell on the Monitor Server.
    -- In PowerShell make sure you can run external scripts, e.g. by issuing the command 
       'set-executionpolicy remotesigned -force', or sign the script.
   
  .EXAMPLE
    cd D:\Monitoring
    .\Record_Device_Usage.ps1
    
  .NOTES
    Prerequisites:
    - Windows PowerShell version 2.0 or higher
    - SQLPS module  (check out http://sev17.com/2010/07/10/making-a-sqlps-module if you need more info)
    This script was tested against SQL 2012, SQL2008 (R2) and SQL2005
       
    2014/10/17 Willem Gossink
#>


### =====  USER DEFINED PARAMETERS  ===== ###

    # Path + name of file with server names whose SQL device usage you want to monitor.
$ServerFile = 'D:\Monitoring\servers.txt'

    # Name of database to be used by this script (will contain table with device usage details). 
    # Database should exist prior to running this script. 
$ManagementDB = 'MonitorDB'

   # Use local machine to store results
   # In case of a SQL named instance, use hard-coded name, e.g.:
   # $SqlHost = server01\sql01
$SqlHost = $env:COMPUTERNAME

   # If $SendMail=1, a mail alert is sent when changes in database device sizes were found, or devices were added/deleted.
$SendMail = 0

### =====  START WORK  ===== ###

Import-Module SQLPS -DisableNameChecking

   # Create holding table
$SQL = "
  IF OBJECT_ID(N'dbo.dbspace', N'U') IS NULL
    CREATE TABLE [dbo].[dbspace] (
        [runid]      [int]
      , [date]       [smalldatetime]
      , [servername] [sysname]
      , [dbname]     [varchar](128)
      , [devname]    [sysname]
      , [type]       [char](4)
      , [filegroup]  [int]
      , [fileid]     [int]
      , [mb_alloc]   [int]
      , [mb_used]    [int]
    ) ; "
Invoke-Sqlcmd ServerInstance $SqlHost -Database $ManagementDB Query "$SQL"

  # Set value for RunID to identify session in holding table
$RunID = (Invoke-Sqlcmd ServerInstance $SqlHost -Database $ManagementDB "SELECT ISNULL(MAX(runid)+1,1) AS 'RunId' FROM dbo.dbspace ") | % { $_.RunID }
$DateTime = Get-Date -UFormat "%Y-%m-%d %H:%M"

  # Determine which server(s) to query
if ($ServerFile) {  
    # retrieve server names from text file
  $ServerList = Import-Csv -Header Server_Name $ServerFile
}
else {
    # or run in standalone mode
  $ServerList = @{Server_Name=$SqlHost}
} 

  # Create list of online databases per server
$ServerDBList = @()
ForEach ($Server in $ServerList | where {$_.Server_Name -ne ''}) {
  $SQL = "SELECT '" +$Server.Server_Name+ "' AS 'Server_Name',
                 [name]                      AS 'Database_Name' 
          FROM   master.sys.databases 
          WHERE  state = 0 ;"
  $ServerDBList += Invoke-Sqlcmd ServerInstance $Server.Server_Name Query "$SQL"
}

  # Record details of log devices per server
$LogSpace = @()
ForEach ($row in $ServerList | where {$_.Server_Name -ne ''}) {
  $SQL = "DBCC sqlperf(logspace) WITH NO_INFOMSGS;"
  $LogSpaceTMP = @()
  $LogSpaceTMP += Invoke-Sqlcmd ServerInstance ($row.Server_Name) Query "$SQL"
  $LogSpaceTMP | Add-Member -type NoteProperty -name Server_Name -Value $row.Server_Name
  $LogSpace += $LogSpaceTMP
} 

  # Record details of data devices per database
$DataSpace = @()
ForEach ($row in $ServerDBList) {
  $SQL = "USE [" + $row.'Database_Name' + "]; DBCC SHOWFILESTATS WITH NO_INFOMSGS;"
  $DataSpaceTMP = @()
  $DataspaceTMP += Invoke-Sqlcmd ServerInstance ($row.Server_Name) Query "$SQL"
  $DataSpaceTMP | Add-Member -type NoteProperty -name Server_Name   -Value $row.Server_Name
  $DataSpaceTMP | Add-Member -type NoteProperty -name Database_Name -Value $row.Database_Name
  $DataSpace += $DataSpaceTMP
}
   
  # Send data for all log devices to holding table on Monitor Server
$SQL=""
ForEach ($row in $LogSpace) {
  $SQL = $SQL +  "INSERT INTO dbo.dbspace values (" +  
                 "$RunID" + ", '" +
                 $DateTime +  "', '" +
                 $row.'Server_Name' + "', '" + 
                 $row.'Database Name' + "', " + 
                 "'log device(s)', " +
                 "'log', " +
                 "0, " +
                 "0, " +
                 ($row.'Log Size (MB)'+1) + ", " + 
                 ((($row.'Log Size (MB)' * $row.'Log Space Used (%)')/100)+1) + "); "  
} 
Invoke-Sqlcmd ServerInstance $SqlHost -Database $ManagementDB  Query "$SQL"

  # Send data for all data devices to holding table on Monitor Server
$SQL=""
ForEach ($row in $DataSpace) {
  $SQL = $SQL +  "INSERT INTO dbo.dbspace values (" +  
                 "$RunID" + ", '" +
                 $DateTime +  "', '" +
                 $row.'Server_Name' + "', '" + 
                 $row.'Database_Name' + "', '" + 
                 $row.'Name' + "', " + 
                 "'data', " +
                 $row.'FileGroup' + ", " + 
                 $row.'FileId' + ", " + 
                 $row.'TotalExtents'*64/1024 + ", " + 
                 $row.'UsedExtents'*64/1024 + "); "
}
Invoke-Sqlcmd ServerInstance $SqlHost -Database $ManagementDB Query "$SQL"

  # Send email alert if this is not the first run and if changes in database device sizes were found 
  # between this run and the previous run.
  # Recipient(s) and sql mail profile are configured in the SQL stored procedure called here.
if ($SendMail -eq 1-and $RunID -gt 1) {
  Invoke-Sqlcmd ServerInstance $SqlHost -Database $ManagementDB  Query "EXEC dbo.Mail_DB_Sizediffs"
}

