-- This script should be run inside of your reporting database

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

SET ANSI_PADDING ON
GO

CREATE TABLE [dbo].[ServerList](
	[ServerID] [tinyint] IDENTITY(1,1) NOT NULL,
	[ServerName] [varchar](20) NOT NULL,
	[IsActive] TINYINT NULL DEFAULT (1)
 CONSTRAINT [PK_ServerList_ServerID] PRIMARY KEY CLUSTERED 
(
	[ServerID] ASC
)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
) ON [PRIMARY]

GO

SET ANSI_PADDING OFF
GO

-- update the name of the MDW database if necessary
INSERT INTO ServerList (ServerName, IsActive)
SELECT 
DISTINCT		instance_name [ServerName], 1 [IsActive]
FROM				MDW.[snapshots].disk_usage d
INNER JOIN	MDW.[core].snapshots_internal sn
ON					sn.snapshot_id = d.snapshot_id
INNER JOIN	MDW.[core].source_info_internal si
ON					si.source_id = sn.source_id ;

GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

SET ANSI_PADDING ON
GO

CREATE TABLE [dbo].[DatabaseProduct](
	[ServerID] [tinyint] NOT NULL,
	[DatabaseName] [nvarchar](128) NOT NULL,
	[ProductType] [varchar](20) NOT NULL,
 CONSTRAINT [PK_DatabaseProduct_ServerID_DatabaseName] PRIMARY KEY CLUSTERED 
(
	[ServerID] ASC,
	[DatabaseName] ASC
)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
) ON [PRIMARY]

GO

SET ANSI_PADDING OFF
GO

ALTER TABLE [dbo].[DatabaseProduct]  WITH CHECK ADD  CONSTRAINT [FK_DatabaseProduct_ServerID] FOREIGN KEY([ServerID])
REFERENCES [dbo].[ServerList] ([ServerID])
GO

ALTER TABLE [dbo].[DatabaseProduct] CHECK CONSTRAINT [FK_DatabaseProduct_ServerID]
GO

SET ANSI_PADDING OFF
GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE TABLE [dbo].[DatabaseProductShared](
	[DatabaseName] [nvarchar](128) NOT NULL
) ON [PRIMARY]

GO

-- change this logic if necessary
INSERT INTO DatabaseProductShared (DatabaseName)
SELECT 'master' [DatabaseName]
UNION ALL
SELECT 'model'
UNION ALL
SELECT 'msdb'
UNION ALL
SELECT 'tempdb'
UNION ALL
SELECT 'distribution'
UNION ALL
SELECT 'ReportServer'
UNION ALL
SELECT 'ReportServerTempDB'
UNION ALL
SELECT 'MDW'
UNION ALL
SELECT 'ServerReporting';

GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE TABLE [dbo].[DiskUsageMart](
	[ServerID] [tinyint] NOT NULL,
	[DatabaseName] [nvarchar](128) NOT NULL,
	[DataSize] [bigint] NULL,
	[IndexSize] [bigint] NULL,
	[LogSize] [bigint] NULL,
	[CaptureDtTm] [smalldatetime] NOT NULL,
 CONSTRAINT [PK_DiskUsageMart_ServerID_DatabaseName_CaptureDtTm] PRIMARY KEY CLUSTERED 
(
	[ServerID] ASC,
	[DatabaseName] ASC,
	[CaptureDtTm] ASC
)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON, DATA_COMPRESSION = PAGE) ON [PRIMARY]
) ON [PRIMARY]

GO

ALTER TABLE [dbo].[DiskUsageMart]  WITH CHECK ADD  CONSTRAINT [FK_DiskUsageMart_ServerID] FOREIGN KEY([ServerID])
REFERENCES [dbo].[ServerList] ([ServerID])
GO

ALTER TABLE [dbo].[DiskUsageMart] CHECK CONSTRAINT [FK_DiskUsageMart_ServerID]
GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE VIEW	[dbo].[vw_DiskUsageDetails]
AS
SELECT			ProductType, 
						ServerName,
						m.DatabaseName,
						'D' AS [DataClass],
						CONVERT(DATE, CaptureDtTm, 101) [CaptureDate],
						SUM(DataSize) AS [Size]
FROM				dbo.DiskUsageMart m
INNER JOIN	dbo.DatabaseProduct p
ON					p.ServerID = m.ServerID
AND					p.DatabaseName = m.DatabaseName
INNER JOIN	dbo.ServerList s
ON					s.ServerID = m.ServerID
GROUP BY	ProductType, 
						ServerName,
						m.DatabaseName,					
						CONVERT(DATE, CaptureDtTm, 101)
UNION ALL
SELECT			ProductType, 
						ServerName,
						m.DatabaseName,
						'I' AS [DataClass],
						CONVERT(DATE, CaptureDtTm, 101) [CaptureDate],
						SUM(IndexSize) AS [Size]
FROM				dbo.DiskUsageMart m
INNER JOIN	dbo.DatabaseProduct p
ON					p.ServerID = m.ServerID
AND					p.DatabaseName = m.DatabaseName
INNER JOIN	dbo.ServerList s
ON					s.ServerID = m.ServerID
GROUP BY	ProductType, 
						ServerName,
						m.DatabaseName,					
						CONVERT(DATE, CaptureDtTm, 101)
UNION ALL
SELECT			ProductType, 
						ServerName,
						m.DatabaseName,
						'L' AS [DataClass],
						CONVERT(DATE, CaptureDtTm, 101) [CaptureDate],
						SUM(LogSize) AS [Size]
FROM				dbo.DiskUsageMart m
INNER JOIN	dbo.DatabaseProduct p
ON					p.ServerID = m.ServerID
AND					p.DatabaseName = m.DatabaseName
INNER JOIN	dbo.ServerList s
ON					s.ServerID = m.ServerID
GROUP BY	ProductType, 
						ServerName,
						m.DatabaseName,					
						CONVERT(DATE, CaptureDtTm, 101);

GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE FUNCTION [dbo].[Pct]
(
@pValue FLOAT,
@pBase FLOAT
)
RETURNS FLOAT
AS
BEGIN

	IF @pBase = 0 RETURN NULL
	
	RETURN @pValue / @pBase
	
END

GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

/*-----------------------------------------------------------------------------------------------------------------------------------------
Description						: Stored procedure used to report product space trends.
Type								: Standard  
Application						: 
Input Parameters			: @pProductType, @pDataClass, @pCurrentDate
Output parameters			: None
Requested By					: 
Requested On					: 
Request Completed On	: 
Request Completed By	: Sean Bowden
Last Modified Date			:
Last Modified By				: 
Update History					: 
-----------------------------------------------------------------------------------------------------------------------------------------*/
CREATE PROCEDURE [dbo].[pr_GetProductSpaceTrendingInfo]
@pProductType VARCHAR(20),
@pDataClass CHAR(1),
@pCurrentDate DATETIME = NULL
AS
BEGIN

	SET NOCOUNT ON;

	DECLARE @currentdate DATETIME
	DECLARE @current DATE
	DECLARE @lastweek DATE
	DECLARE @lastmonth DATE
	DECLARE @sixmonth DATE
	DECLARE @lastyear DATE
	
	IF @pCurrentDate IS NULL
		SET @currentdate = CAST(GETDATE() AS DATE)
	ELSE
		SET @currentdate = CAST(@pCurrentDate AS DATE)

	-- these will be our exact or approximate capture dates for the relevant reporting periods
	SET @current = @currentdate - (DATEPART(dw,@currentdate)-1);
	SET @lastweek = DATEADD(wk,-1,@current);
	SET @lastmonth = DATEADD(m,-1,@current);
	SET @sixmonth = DATEADD(m,-6,@current);
	SET @lastyear = DATEADD(yy,-1,@current)

	;WITH CurrentData
	AS
	(
	SELECT 
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = @current
	AND (ProductType = @pProductType OR @pProductType = 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	UNION ALL
	SELECT 
	'All Other Products', 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = @current
	AND (ProductType <> @pProductType AND @pProductType <> 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	),
	LastWeekData
	AS
	(
	SELECT 
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = @lastweek
	AND (ProductType = @pProductType OR @pProductType = 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	UNION ALL
	SELECT 
	'All Other Products', 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = @lastweek
	AND (ProductType <> @pProductType AND @pProductType <> 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	),
	LastMonthData
	AS
	(
	SELECT 
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @lastmonth
	)
	AND (ProductType = @pProductType OR @pProductType = 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	UNION ALL
	SELECT 
	'All Other Products', 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @lastmonth
	)
	AND (ProductType <> @pProductType AND @pProductType <> 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	),
	SixMonthData
	AS
	(
	SELECT 
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @sixmonth
	)
	AND (ProductType = @pProductType OR @pProductType = 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	UNION ALL
	SELECT 
	'All Other Products', 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @sixmonth
	)
	AND (ProductType <> @pProductType AND @pProductType <> 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	),
	LastYearData
	AS
	(
	SELECT 
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @lastyear
	)
	AND (ProductType = @pProductType OR @pProductType = 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	ProductType, 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	UNION ALL
	SELECT 
	'All Other Products', 
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END DataClass, 
	SUM(Size) [Value]
	FROM vw_DiskUsageDetails
	WHERE CaptureDate = 
	(
		SELECT MIN(CaptureDate) 
		FROM vw_DiskUsageDetails 
		WHERE CaptureDate >= @lastyear
	)
	AND (ProductType <> @pProductType AND @pProductType <> 'All Products')
	AND (DataClass = @pDataClass OR @pDataClass = 'Z')
	GROUP BY
	CASE
		WHEN DataClass = 'D' THEN 'Data'
		WHEN DataClass = 'I' THEN 'Index'
		WHEN DataClass = 'L' THEN 'Log'
	END
	)

	SELECT 
	a.ProductType, 
	a.DataClass, 
	a.Value [CurrentValue], 
	b.Value [LastWeekValue], 
	dbo.Pct(a.Value-b.Value,b.Value) [LastWeekPctChange],
	c.Value [LastMonthValue], 
	dbo.Pct(a.Value-c.Value,c.Value) [LastMonthPctChange],
	d.Value [SixMonthValue], 
	dbo.Pct(a.Value-d.Value,d.Value) [SixMonthPctChange],
	e.Value [LastYearValue], 
	dbo.Pct(a.Value-e.Value,e.Value) [LastYearPctChange]
	FROM CurrentData a
	LEFT JOIN LastWeekData b
	ON a.DataClass = b.DataClass
	AND a.ProductType = b.ProductType
	LEFT JOIN LastMonthData c
	ON a.DataClass = c.DataClass
	AND a.ProductType = c.ProductType
	LEFT JOIN SixMonthData d
	ON a.DataClass = d.DataClass
	AND a.ProductType = d.ProductType
	LEFT JOIN LastYearData e
	ON a.DataClass = e.DataClass
	AND a.ProductType = e.ProductType;
	
END

GO