USE [MapDemo]
GO

--#############################################################################
-- Valid greometry shape data, and convert them to geography data.
--#############################################################################

UPDATE  tl_2011_us_state
SET     Geog = GEOGRAPHY::STGeomFromWKB(Geom.MakeValid().STAsBinary(), 4326)
WHERE   NAME NOT IN('Massachusetts', 'New Mexico', 'Wyoming', 'Montana', 'Texas', 'North Carolina', 'Arizona', 'Tennessee', 'Utah');

UPDATE  tl_2011_us_state
SET     Geog = GEOGRAPHY::STGeomFromWKB(Geom.STUnion(Geom.STStartPoint()).MakeValid().STAsBinary(), 4326)
WHERE   NAME = 'Massachusetts'

UPDATE  tl_2011_us_state
SET     Geog = GEOGRAPHY::STGeomFromWKB(Geom.STBuffer(0.00001).STBuffer(-0.00001).MakeValid().STAsBinary(), 4326)
WHERE   NAME IN ('New Mexico', 'Wyoming', 'North Carolina', 'Tennessee')

UPDATE  tl_2011_us_state
SET     Geog = GEOGRAPHY::STGeomFromWKB(Geom.Reduce(0.00001).MakeValid().STAsBinary(), 4326)
WHERE   NAME IN ('Texas', 'Montana', 'Arizona', 'Utah')

--#############################################################################
-- Verify and recreate production table dbo.[StateMap]
--#############################################################################

IF OBJECT_ID (N'dbo.[StateMap]', N'U') IS NOT NULL
    DROP TABLE dbo.[StateMap];
GO

CREATE TABLE dbo.[StateMap] 
(
    ID INT IDENTITY(1, 1) NOT NULL,
    StateFP VARCHAR(2) NOT NULL,
    StateUSPS  VARCHAR(2) NOT NULL, 
    StateName  VARCHAR(50) NOT NULL, 
    Geog  GEOGRAPHY NOT NULL,
    MapSetId INT NOT NULL
    CONSTRAINT PK_State PRIMARY KEY (ID) 
); 

--#############################################################################
-- Insert map data into dbo.[StateMap]
--#############################################################################

INSERT  dbo.[StateMap] 
SELECT  STATEFP AS StateFP,
        STUSPS AS StateUSPS,
        NAME AS StateName,
        Geog.MakeValid().Reduce(250) AS Geog,
        1 AS MapSetId
FROM    dbo.tl_2011_us_state
WHERE STUSPS NOT IN ('AS', 'MP', 'GU');

CREATE SPATIAL INDEX US_States ON dbo.[StateMap](Geog);

--#############################################################################
-- Reposition Alaska
--#############################################################################

DECLARE @StateFP VARCHAR(10)
DECLARE @Geog GEOGRAPHY
DECLARE @Geom GEOMETRY

SELECT  @Geog = Geog
FROM    dbo.[StateMap]
WHERE   StateUSPS = 'AK'
        
SET @Geom = GEOMETRY::STGeomFromWKB(@Geog.STAsBinary(), 4326)
SET @Geom = AffineTransform::Scale(.15, .25).Apply(@Geom)
SET @Geom = AffineTransform::Translate(-93, 13).Apply(@Geom) 

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::STGeomFromWKB(@Geom.MakeValid().STAsBinary(), 4326)
WHERE   StateUSPS = 'AK'

--#############################################################################
-- Reposition Hawaii
--#############################################################################

SELECT  @Geog = Geog
FROM    dbo.[StateMap]
WHERE   StateUSPS = 'HI'
        
SET @Geom = GEOMETRY::STGeomFromWKB(@Geog.STAsBinary(), 4326)
SET @Geom = AffineTransform::Scale(1.25, 1.25).Apply(@Geom)
SET @Geom = AffineTransform::Translate(89, 2.5).Apply(@Geom)

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::STGeomFromWKB(@Geom.MakeValid().STAsBinary(), 4326)
WHERE   StateUSPS = 'HI'

--#############################################################################
-- Reposition Puerto Rico
--#############################################################################

SELECT  @Geog = Geog
FROM    dbo.[StateMap]
WHERE   StateUSPS = 'PR'
        
SET @Geom = GEOMETRY::STGeomFromWKB(@Geog.STAsBinary(), 4326)
SET @Geom = AffineTransform::Scale(1.35, 1.35).Apply(@Geom)
SET @Geom = AffineTransform::Translate(20, -2.5).Apply(@Geom)

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::STGeomFromWKB(@Geom.MakeValid().STAsBinary(), 4326)
WHERE   StateUSPS = 'PR'

--#############################################################################
-- Reposition Virgin Island
--#############################################################################

SELECT  @Geog = Geog
FROM    dbo.[StateMap]
WHERE   StateUSPS = 'VI'

SET @Geom = GEOMETRY::STGeomFromWKB(@Geog.STAsBinary(), 4326)
SET @Geom = AffineTransform::Scale(1.35, 1.35).Apply(@Geom)
SET @Geom = AffineTransform::Translate(20, -2.5).Apply(@Geom)

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::STGeomFromWKB(@Geom.MakeValid().STAsBinary(), 4326)
WHERE   StateUSPS = 'VI'

--#############################################################################
-- Filter Northwestern Hawaiian Islands 
--#############################################################################

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::Parse('POINT(-107 28)').STBuffer(500000).STIntersection(Geog)
WHERE   StateUSPS = 'HI'

--#############################################################################
-- Filter unwanted small islands in Alaska
--#############################################################################

UPDATE  dbo.[StateMap]
SET     Geog = GEOGRAPHY::Parse('POINT(-117 28)').STBuffer(500000).STIntersection(Geog)
WHERE   StateUSPS = 'AK'
