
IF (EXISTS(SELECT * FROM sys.objects WHERE [object_id] = OBJECT_ID(N'[dbo].[fn_LastDay]') AND ([type]='IF' OR [type]='FN' OR [type]='TF')))
DROP FUNCTION [dbo].[fn_LastDay]
GO

SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO
CREATE FUNCTION [dbo].[fn_LastDay]
(
  @YearIn        int,
  @MonthIn       int,
  @DayOfHoliday  int,
  @FullWeekFlag  bit
)
/*
  
   @YearIn        - Year Of Holiday Check
   @MonthIn       - Month Of Holiday Check
   @DayOfHoliday  - Day Of Holiday (0 to 6) Sun to Sat
   @FullWeekFlag  - Based on full week (Starts counting with first sunday)
  
   ex: Last Monday in may = LastDay(2004, 5, 1, 0)
  
*/
  RETURNS int
  AS
  BEGIN
    DECLARE @ReturnDay          int
    DECLARE @LastDayOfMonth     datetime
    DECLARE @LastDayOfWeek      int --(0 - 6) Sun to Sat
    DECLARE @LastDayDifference  int
    
    /*
      
      Calclulate the last day of the month based on the Year, Month, and Day passed in.
      Adjust date for Year and Month wrapping.
      
    */
    IF @MonthIn + 1 > 12
      BEGIN
        SELECT
          @LastDayOfMonth = DATEADD(dd, -1, CAST(CAST(@YearIn + 1 AS char(4)) + '-01-01' AS datetime))
      END
    ELSE
      BEGIN
        SELECT
          @LastDayOfMonth = DATEADD(dd, -1, CAST(CAST(@YearIn AS char(4)) + '-' + CAST((@MonthIn + 1) AS char(2)) + '-01' AS datetime))
      END
    /*
      
      Find the name day of the last day of the month.  Convert to a number (0 - 6) for calculations.
      
    */
    SELECT
      @LastDayOfWeek = DATEPART(dw, @LastDayOfMonth) - 1
    /*
      
      Find the number of days different from the last name day of the month and the name day of the holiday.
      
    */
    SELECT
      @LastDayDifference = (((@LastDayOfWeek - @DayOfHoliday) + 7) % 7) * -1
    /*
      
      Adjust the number day of the holiday based on the last day of the month - the difference.
      
    */
    SELECT
      @ReturnDay = DATEPART(dd, DATEADD(dd, @LastDayDifference, @LastDayOfMonth))
    /*
      
      if the FullWeekFlag is set (based on last saturday in month) then adjust the ReturnDay if needed.
      
    */
    IF
      @FullWeekFlag = 1
      AND @LastDayOfWeek < 6
      AND @LastDayOfWeek >=
        (
          DATEPART
          (
            dw,
            CAST
            (
              CAST(@YearIn AS char(4)) +
              '-' +
              CAST(@MonthIn AS char(2)) +
              '-' +
              CAST(@ReturnDay AS char(2))
              AS datetime
            )
          ) - 1
        )
      BEGIN
        SELECT
          @ReturnDay = @ReturnDay - 7
      END
    /*
      
      Return the number date within the month that the holiday falls on for the specified year.
      
    */
    RETURN(@ReturnDay)
 END
GO

