
IF (EXISTS(SELECT * FROM sys.objects WHERE [object_id] = OBJECT_ID(N'[dbo].[fn_FindDate]') AND ([type]='IF' OR [type]='FN' OR [type]='TF')))
DROP FUNCTION [dbo].[fn_FindDate]
GO

SET ANSI_NULLS ON
SET QUOTED_IDENTIFIER ON
GO

CREATE FUNCTION [dbo].[fn_FindDate]
(
  @HolidayIDIn     int,
  @TitleIn         varchar(100),
  @ReportFlagIn    bit,
  @DayOffFlagIn    bit,
  @StartYearIn     int,
  @EndYearIn       int,
  @DaysPriorIn     int,
  @DaysAfterIn     int,
  @MonthOfIn       int,
  @DayOfIn         int,
  @NameDayIn       int,
  @WeekNumberIn    int,
  @LastDayFlagIn   bit,
  @FullWeekFlagIn  bit,
  @BaseHolidayIn   int,
  @DaysVarianceIn  int,
  @SpecialCalcIn   int,
  @YearIn          int,
  @SlideDateIn     int
)
/*
  
  Any fields not used in calculating a date can be defaulted to 0 (Not used).
  
  @HolidayIDIn     - Unique ID for a specific holiday.
  @TitleIn         - Title of a holiday.
  @ReportFlagIn    - Should this be reported (0, 1).
  @DayOffFlagIn    - Day off from work flag (0, 1).
  @StartYearIn     - Year the holiday started.
  @EndYearIn       - Year the holiday finished.
  @DaysPriorIn     - Days prior to the date recorded the holiday starts.
  @DaysAfterIn     - Days after to the date recorded the holiday continues.
  @MonthOfIn       - Month the holiday is in.
  @DayOfIn         - Number day the holiday starts.
  @NameDayIn       - Name day (Sun - Sat) as a number the holiday starts.
  @WeekNumberIn    - Week of the month the holiday starts.
  @LastDayFlagIn   - Does the holiday start on the last day of a specific name day.
  @FullWeekFlagIn  - Does the holiday use full weeks only to determine the start day.
  @BaseHolidayIn   - What holiday is used as the basis for this holiday.
  @DaysVarianceIn  - Number of days holiday is offset from the base holiday.
  @SpecialCalcIn   - Which special calculation is used to find the holiday date.
  @YearIn          - Year to check to determine the holiday date.
  @SlideDateIn     - Does this holiday change the reported date if it falls in the weekend.
    
*/
  RETURNS datetime
  AS
  BEGIN
    
    DECLARE @DateType   int
    DECLARE @DateOut    datetime
    DECLARE @DateCheck  datetime
    DECLARE @HoldID     int
    DECLARE @NewMonth   int
    DECLARE @NumYears   int
    DECLARE @DayOfWeek  int
    
    IF EXISTS(SELECT 1 FROM HolidayDates WHERE HolidayID = @HolidayIDIn AND Year = @YearIn)
      BEGIN
        /*
          
          Retrieve date from the HolidayDates table.  Much faster than recalculating.
          
        */
        
        SELECT
          @DateOut = DayStart
        FROM HolidayDates
        WHERE
          HolidayID = @HolidayIDIn
          AND Year = @YearIn
      END
    ELSE
      BEGIN
        /*
          
          Find what type (DateType) of holiday this is.
            0 - Not Set.
            1 - Specific Number Day Of Month.
            2 - Specified By Name Day Of Month Or Specific Week Of Month.
            3 - Uses A Different Holiday As Its Base.
            4 - Uses A Special Calculation (Complex Calculation).
          
        */
        SELECT
          @DateType = 0,
          @DateOut  = NULL
        
        IF @DayOfIn > 0
          BEGIN
            SELECT
              @DateType = 1
          END
        IF @NameDayIn > 0 OR @WeekNumberIn > 0
          BEGIN
            SELECT
              @DateType = 2
          END
        IF @BaseHolidayIn > 0
          BEGIN
            SELECT
              @DateType = 3
          END
        IF @SpecialCalcIn > 0
          BEGIN
            SELECT
              @DateType = 4
          END
        
        IF @DateType = 1
          BEGIN
            /*
              
              Calclulate the specific date based on the Year, Month, and Day passed in.
              Adjust date for Year and Month wrapping.
              
            */
            IF @MonthOfIn > 12
              BEGIN
                /*
                  
                  Find the last day of the month for the specified date as months will vary on the number of days (28 - 31).
                  If the last day of the month for the specified month is less than the specified day, then use the
                  last day of the month as the specified day otherwise calculate the specified day.
                  
                */
                SELECT
                  @NumYears = CAST((@MonthOfIn / 12) AS int),
                  @NewMonth = @MonthOfIn % 12
                SELECT
                  @DateCheck =
                    DATEADD
                    (
                      dd,
                      -1,
                      CAST
                      (
                        CAST((@YearIn + @NumYears) AS char(4)) +
                        '-' +
                        CAST((@NewMonth + 1) AS char(2)) +
                        '-01'
                        AS datetime
                      )
                    )
                IF DATEPART(dd, @DateCheck) < @DayOfIn
                  BEGIN
                    SELECT
                      @DateOut = @DateCheck
                  END
                ELSE
                  BEGIN
                    SELECT
                      @DateOut =
                        CAST
                        (
                          CAST((@YearIn + @NumYears) AS char(4)) +
                          '-' +
                          CAST(@NewMonth AS char(2)) +
                          '-' +
                          CAST(@DayOfIn AS char(2))
                          AS datetime
                        )
                  END
              END
            ELSE
              BEGIN
                /*
                  
                  Find the last day of the month for the specified date as months will vary on the number of days (28 - 31).
                  If the last day of the month for the specified month is less than the specified day, then use the
                  last day of the month as the specified day otherwise calculate the specified day.
                  
                */
                IF @MonthOfIn < 12
                  BEGIN
                    SELECT
                      @DateCheck =
                        DATEADD
                        (
                          dd,
                          -1,
                          CAST
                          (
                            CAST((@YearIn) AS char(4)) +
                            '-' +
                            CAST((@MonthOfIn + 1) AS char(2))
                            + '-01'
                            AS datetime
                          )
                        )
                  END
                ELSE
                  BEGIN
                    SELECT
                      @DateCheck = CAST(CAST((@YearIn) AS char(4)) + '-12-31' AS datetime)
                  END
                IF DATEPART(dd, @DateCheck) < @DayOfIn
                  BEGIN
                    SELECT
                      @DateOut = @DateCheck
                  END
                ELSE
                  BEGIN
                    SELECT
                      @DateOut =
                        CAST
                        (
                          CAST(@YearIn AS char(4)) +
                          '-' +
                          CAST(@MonthOfIn AS char(2)) +
                          '-' +
                          CAST(@DayOfIn AS char(2))
                          AS datetime
                        )
                  END
              END
          END
        IF @DateType = 2
          BEGIN
            /*
              
              Uses a specific name day or week of month for the date.
              If the LastDayFlag is on, then need to find the final occurrance of the name day
              or the last week of the month.  Both of these use a function to find the specific
              number day of the month.
              
            */
            IF @LastDayFlagIn = 1
              BEGIN
                SELECT
                  @DateOut =
                    CAST
                    (
                      CAST(@YearIn AS char(4)) +
                      '-' +
                      CAST(@MonthOfIn AS char(2)) +
                      '-' +
                      CAST
                      (
                        dbo.fn_LastDay
                        (
                          @YearIn,
                          @MonthOfIn,
                          @NameDayIn - 1,
                          @FullWeekFlagIn
                        )
                        AS char(2)
                      )
                      AS datetime
                    )
              END
            ELSE
              BEGIN
                SELECT
                  @DateOut =
                    CAST
                    (
                      CAST(@YearIn AS char(4)) +
                      '-' +
                      CAST(@MonthOfIn AS char(2)) +
                      '-' +
                      CAST
                      (
                        dbo.fn_WhatDate
                        (
                          @YearIn,
                          @MonthOfIn,
                          @NameDayIn - 1,
                          @WeekNumberIn,
                          @FullWeekFlagIn
                        )
                        AS char(2)
                      )
                      AS datetime
                    )
              END
          END
        IF @DateType = 3
          BEGIN
            /*
              
              A different holiday is used as the basis for this date.  This date will be a variance (Number
              of days offset) from the base holiday.
              
            */
            SELECT
              @DateOut =
                dbo.fn_FindDate
                (
                  HolidayID,
                  Title,
                  ReportFlag,
                  DayOffFlag,
                  StartYear,
                  EndYear,
                  DaysPrior,
                  DaysAfter,
                  MonthOf,
                  DayOf,
                  NameDay,
                  WeekNumber,
                  LastDayFlag,
                  FullWeekFlag,
                  BaseHoliday,
                  DaysVariance,
                  SpecialCalc,
                  @YearIn,
                  SlideDate
                ) +
                @DaysVarianceIn
            FROM Holidays
            WHERE
              HolidayID = @BaseHolidayIn
          END
        IF @DateType = 4
          BEGIN
            /*
              
              A special complex calculation is used to find the date.
              
            */
            IF @SpecialCalcIn = 1  -- Get Easter Day
              BEGIN
                /*
                  
                  Easter uses a function with the calculations to find the correct date for a specific year.
                  
                */
                SELECT
                  @DateOut = dbo.fn_GetEaster(@YearIn)
              END
            IF @SpecialCalcIn = 2  -- Get Election Day
              BEGIN
                /*
                  
                  Election Day happens once every 4 years on the first tuesday after the first monday.
                  If fn_WhatDate() returns (1), then add 7 because that means November started on tuesday
                  and we need to be after the first monday.
                  
                */
                SELECT
                  @DayOfWeek = dbo.fn_WhatDate(CAST(((@YearIn + 3) / 4) AS int) * 4, 11, 2, 1, 0)
                IF @DayOfWeek = 1
                  BEGIN
                    SELECT
                      @DateOut =
                        CAST
                        (
                          CAST
                          (
                            CAST(((@YearIn + 3) / 4) AS int) * 4
                            AS char(4)
                          ) +
                          '-11-' +
                          CAST((@DayOfWeek + 7) AS char(2))
                          AS datetime
                        )
                  END
                ELSE
                  BEGIN
                    SELECT
                      @DateOut =
                        CAST
                        (
                          CAST
                          (
                            CAST(((@YearIn + 3) / 4) AS int) * 4
                            AS char(4)
                          ) +
                          '-11-' +
                          CAST((@DayOfWeek + 7) AS char(2))
                          AS datetime
                        )
                  END
              END
            IF @SpecialCalcIn = 3  -- Get Federal Income Taxes, Due on 4-15 or monday(3rd) after if it falls on a weekend.
              BEGIN
                /*
                  
                  Income taxes are due on the same day every year except when it falls on a weekend.
                  On a weekend, the income taxes are due on the following monday.
                  
                  This special calculation is now obsolete with the addition of the SlideDate field.  Now a regular
                  Holiday record can be created for April 15 with a SlideDate set to 3 (Following Monday If Weekend).
                  
                */
                SELECT
                  @DateOut = CAST(CAST(@YearIn AS char(4)) + '-04-15' AS datetime)
                IF DATEPART(dw, @DateOut) = 1 OR DATEPART(dw, @DateOut) = 7
                  BEGIN
                    SELECT
                      @DateOut =
                        CAST
                        (
                          CAST(@YearIn AS char(4)) +
                          '-04-' +
                          CAST(dbo.fn_WhatDate(@YearIn, 4, 1, 3, 0) AS char(2))
                          AS datetime
                        )
                  END
              END
            IF @SpecialCalcIn = 4  -- Get Inauguration Day (on 1-20 once every 4 years)
              BEGIN
                /*
                  
                  Presidents are inaugurated only once every 4 years (4 year term).
                  
                */
                SELECT
                  @DateOut =
                    CAST
                    (
                      CAST
                      (
                        (
                          CAST(((@YearIn + 2) / 4) AS int)
                          * 4 + 1
                        )
                        AS char(4)
                      ) +
                      '-01-20'
                      AS datetime
                    )
              END
          END
        IF @SlideDateIn > 0
          BEGIN
            /*
              
              Some special dates need to be adjusted if they fall on a weekend.  For this, there are
              3 different ways to adjust the date.
                1 - Shift to the closest weekend (Sat = Fri, Sun = Mon).
                2 - Shift to Friday (Sat, Sun = Fri).
                3 - Shift to Monday (Sat, Sun = Mon).
              
            */
            SELECT
              @DayOfWeek = DATEPART(dw, @DateOut)
            IF @DayOfWeek = 1 OR @DayOfWeek = 7
              BEGIN
                SELECT
                  @DateOut =
                    (
                      CASE @SlideDateIn
                        WHEN 1 THEN  -- Shift To Closest Weekend
                          (
                            CASE @DayOfWeek
                              WHEN 1 THEN  -- Sunday
                                DATEADD(dd, 1, @DateOut)
                              WHEN 7 THEN  -- Saturday
                                DATEADD(dd, -1, @DateOut)
                            END
                          )
                        WHEN 2 THEN  -- Shift To Friday
                          (
                            CASE @DayOfWeek
                              WHEN 1 THEN  -- Sunday
                                DATEADD(dd, -2, @DateOut)
                              WHEN 7 THEN  -- Saturday
                                DATEADD(dd, -1, @DateOut)
                            END
                          )
                        WHEN 3 THEN  -- Shift To Monday
                          (
                            CASE @DayOfWeek
                              WHEN 1 THEN  -- Sunday
                                DATEADD(dd, 1, @DateOut)
                              WHEN 7 THEN  -- Saturday
                                DATEADD(dd, 2, @DateOut)
                            END
                          )
                      END
                    )
              END
          END
      END
    /*
      
      Return the calculated date, with any adjustments, for the special day of the specified year.
      
    */
    RETURN(@DateOut)
 END
GO

