/*
	SCRIPT:  05_GenerateSupportingProcs.sql

	DESCRIPTION
	===========
	Generate stored procedures to carry out the required candidate key assessment.
	
*/
DECLARE 
	@DBName SYSNAME = DB_NAME()
RAISERROR('DATABASE: %s, SCRIPT: 05_GenerateSupportingProcs.sql',10,1,@DBName) WITH NOWAIT
PRINT '=============================================================='
GO
-------------------------------------------------------------------------------------------------
DECLARE @SQL VARCHAR(MAX)
DECLARE @CRLF CHAR(2)=CHAR(13)+CHAR(10)
SELECT @SQL = COALESCE(@SQL+';'+@CRLF,'')
+	'DROP '
+	CASE type
		WHEN 'FN' THEN 'FUNCTION '
		WHEN 'P' THEN 'PROC '
	END
+ OBJECT_SCHEMA_NAME(object_id)+'.'+OBJECT_NAME(object_id)
FROM sys.objects
WHERE type IN ('P')
AND OBJECT_SCHEMA_NAME(object_id) IN ('Validate','CandidateKey')
ORDER BY type DESC
PRINT @SQL
EXEC(@SQL)
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.FlushValidateSchema
AS
SET NOCOUNT ON

DECLARE @ValidateSchemaObjects TABLE (TableID INT NOT NULL)
INSERT INTO @ValidateSchemaObjects (TableID)
SELECT object_id
FROM sys.objects AS O
WHERE O.schema_id=SCHEMA_ID('Validate')
AND O.name LIKE 'T[0-9][0-9][0-9][0-9]%'
ORDER BY object_id


DECLARE @TableID INT
DECLARE @SQL  VARCHAR(MAX)
SET @TableID = 0
WHILE @TableID IS NOT NULL
	BEGIN
		SELECT @TableID=MIN(TableID)
		FROM @ValidateSchemaObjects
		WHERE TableID>@TableID

		IF @TableID IS NOT NULL
			BEGIN
				SET @SQL = 'DROP TABLE '
					+	QUOTENAME(OBJECT_SCHEMA_NAME(@TableID))
					+	'.'
					+	QUOTENAME(OBJECT_NAME(@TableID))	
				PRINT @SQL
				EXEC (@SQL)
			END
	END
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.FlushValidateSchema'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.AssessCandidateUniqueness
	@TableID INT
AS
SET NOCOUNT ON

DECLARE @SQL VARCHAR(MAX)
SELECT @SQL=COALESCE(@SQL+',','')
+	'CAST(CASE WHEN COUNT(*)>0 AND COUNT(*)-COUNT(DISTINCT '
+	ColumnName
+	') =0 THEN 1 ELSE 0 END AS BIT) AS '
+	ColumnName
FROM CandidateKey.CandidateUniqueConstraints
WHERE TableID=@TableID
ORDER BY KeyPriority

SET @SQL='SELECT COUNT(*) AS TotalRows,' + @SQL
+	' INTO Validate.T'
+	CAST(@TableID AS SYSNAME)
+	' FROM '
+	OBJECT_SCHEMA_NAME(@TableID)+'.'+OBJECT_NAME(@TableID)

EXEC(@SQL)
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.AssessCandidateUniqueness'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.ExecuteCandidateUniqueness
	@ThresholdFloorRowCount BIGINT=0, --##PARAM @ThresholdFloorRowCount The minimum number of records for which a uniqueness assessment will take place.
	@ThresholdCeilingRowCount BIGINT=10000 --##PARAM @@ThresholdCeilingRowCount The maximum number of records for which a uniqueness assessment will take place.
AS
SET NOCOUNT ON

DECLARE 
	@TableID INT,
	@RowCount BIGINT,
	@FQTableName SYSNAME
SET @TableID=0
WHILE @TableID IS NOT NULL
	BEGIN
		SELECT @TableID=MIN(TableID)
		FROM CandidateKey.CandidateUniqueConstraints
		WHERE TableID>@TableID

		IF @TableID IS NOT NULL
			BEGIN
				SET @FQTableName=QUOTENAME(OBJECT_SCHEMA_NAME(@TableID))+'.'+QUOTENAME(OBJECT_NAME(@TableID))
				SELECT @RowCount=SUM(rows)
				FROM sys.partitions
				WHERE object_id = @TableID
				AND index_id IN(0,1)
				
				IF @RowCount > @ThresholdCeilingRowCount
					BEGIN
						RAISERROR('THRESHOLD %I64d EXCEEDED: %s = %I64d ',10,1,@ThresholdCeilingRowCount,@FQTableName,@RowCount) WITH NOWAIT
					END
				ELSE
					BEGIN
						IF @RowCount >= @ThresholdFloorRowCount
							exec CandidateKey.AssessCandidateUniqueness @TableID
					END
			END
	END
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.ExecuteCandidateUniqueness'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GetCandidateKeys
AS
SET NOCOUNT ON

TRUNCATE TABLE CandidateKey.CandidateUniqueConstraints

-- IDENTITY COLUMNS
INSERT INTO CandidateKey.CandidateUniqueConstraints(TableID, KeyPriority, Is_Nullable,ColumnName)
SELECT 
	TableId=object_id,
	KeyPriority=1 ,
	Is_Nullable,
	ColumnName=name
FROM sys.columns
WHERE is_identity=1
AND OBJECTPROPERTY(object_id,'IsUserTable')=1

RAISERROR('Candidate Keys with an identity property %d',10,1,@@ROWCOUNT) WITH NOWAIT

-- SEQUENCE COLUMNS
INSERT INTO CandidateKey.CandidateUniqueConstraints(TableID, KeyPriority, Is_Nullable,ColumnName)
SELECT 
	TableId=c.object_id,
	KeyPriority=2 ,
	C.Is_Nullable,
	ColumnName=c.name
FROM sys.sequences AS SQ
	INNER JOIN sysdepends DP
	ON SQ.object_id=DP.depid
	INNER JOIN sys.columns AS C
	ON C.default_object_id = DP.id
WHERE OBJECTPROPERTY(C.object_id,'IsUserTable')=1
RAISERROR('Candidate Keys using a SEQUENCE %d',10,1,@@ROWCOUNT) WITH NOWAIT


-- Other Candidates
INSERT INTO CandidateKey.CandidateUniqueConstraints(TableID, KeyPriority, Is_Nullable,ColumnName)
SELECT 
	SRC.TableId,
	SRC.KeyPriority,
	SRC.Is_Nullable,
	SRC.ColumnName
FROM CandidateKey.CandidatePrimaryKeys AS SRC
	LEFT JOIN CandidateKey.CandidateUniqueConstraints AS TARG
	ON SRC.TableID = TARG.TableID
	AND SRC.ColumnName = TARG.ColumnName
WHERE TARG.TableID IS NULL

RAISERROR('Candidate Keys with column properties  %d',10,1,@@ROWCOUNT) WITH NOWAIT

GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GetCandidateKeys'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.ExtractValidatedCandidates
	@TableName SYSNAME -- The name of the Validate.T<Object_id> table
AS
SET NOCOUNT ON

--Must be a table matching the desired pattern in the Validate schema.
IF @TableName NOT LIKE 'T[0-9][0-9][0-9]%' OR NOT EXISTS (SELECT 1 FROM sys.objects WHERE name = @TableName AND schema_id=SCHEMA_ID('Validate'))
	BEGIN
		RAISERROR('Table %s is not a validation table',16,1,@TableName) with nowait
		RETURN 1
	END


DECLARE 
	@TableID VARCHAR(10),
	@ValidationTableID INT ,
	@ColumnList VARCHAR(MAX),
	@SQL VARCHAR(MAX)

SET @TableID = RIGHT(@TableName,LEN(@TableName)-1) --The numeric part of the Validate.T<Object_id> table name
SET @ValidationTableID = OBJECT_ID('Validate.'+@TableName) -- The object_id of the actual Validate.T<Object_id> table name


SELECT @ColumnList=COALESCE(@ColumnList+',','')
+	name
FROM sys.columns AS C
WHERE object_id=@ValidationTableID
AND name<>'TotalRows'
ORDER BY C.column_id

SET @SQL='INSERT INTO Validate.CandidateUniqueConstraints(TableId, ColumnName, IsValidated)
SELECT TableId, ColumnName, IsValidated
FROM (
	SELECT '
+	@TableID 
+	' AS TableId, '
+	@ColumnList
+	' FROM Validate.'+@TableName+') p
	UNPIVOT (
		IsValidated FOR ColumnName IN ('
+	@ColumnList
+	')	)AS unpvt'

print @TableName
EXEC(@SQL)
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.ExtractValidatedCandidates'
GO

-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.SetCandidateValidity
AS
SET NOCOUNT ON

DECLARE @TableName SYSNAME='',
	@SchemaID int

SET @SchemaID=SCHEMA_ID('Validate')

TRUNCATE TABLE Validate.CandidateUniqueConstraints

WHILE @TableName IS NOT NULL
	BEGIN
		SELECT @TableName = MIN(name)
		FROM sys.objects
		WHERE name>@TableName
		AND schema_id=@SchemaID
		AND type='U'
		AND name LIKE 'T[0-9][0-9][0-9][0-9]%'

		IF @TableName IS NOT NULL
			BEGIN
				EXEC CandidateKey.ExtractValidatedCandidates @TableName
			END
	END

UPDATE DEST
SET DEST.IsValidated=SRC.IsValidated
FROM	Validate.CandidateUniqueConstraints AS SRC
	INNER JOIN CandidateKey.CandidateUniqueConstraints AS DEST
	ON	SRC.TableID = DEST.TableID
	AND SRC.ColumnName = DEST.ColumnName
WHERE SRC.IsValidated<>DEST.IsValidated

RAISERROR ('%d Candidate Keys Updated',10,1,@@ROWCOUNT) WITH NOWAIT

-- Discard any candidates that didn't resolve as PK Candidates
DELETE FROM CandidateKey.CandidateUniqueConstraints WHERE IsValidated=0
RAISERROR ('%d Invalid candidate Keys removed',10,1,@@ROWCOUNT) WITH NOWAIT

GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.SetCandidateValidity'
GO
-------------------------------------------------------------------------------------------------
IF EXISTS(SELECT * FROM sys.objects WHERE type='P' AND name='GetCandidateForeignKey' AND schema_id=SCHEMA_ID('CandidateKey'))
	BEGIN
		DROP PROC CandidateKey.GetCandidateForeignKey
		PRINT 'PROC DROPPED: CandidateKey.GetCandidateForeignKey'
	END
GO
CREATE PROC CandidateKey.GetCandidateForeignKey
	@TableID INT -- The object_id of the parent object
AS
SET NOCOUNT ON
DECLARE 
	@PKFieldCount int,
	@SQLReferences VARCHAR(MAX)	


DECLARE @FKTable AS TABLE (
	Object_id INT NOT NULL,
	ColumnId INT NOT NULL,
	StandardisedObjectName SYSNAME NOT NULL ,
	OriginalColumnName SYSNAME NOT NULL ,
	StandardisedColumnName SYSNAME NOT NULL
)


-- Retrieve the number of fields in the primary key for the table.
SELECT @PKFieldCount = PKFieldCount
FROM CandidateKey.PKFieldCount
WHERE Object_ID = @TableID

-- Build a comma delimited list of fields in the primary key table.
SELECT @SQLReferences = COALESCE(@SQLReferences+',','')+ OriginalColumnName
FROM 	CandidateKey.StandardisedPKItems
WHERE Object_Id=@TableID
ORDER BY ColumnId

SET @SQLReferences = ' REFERENCES '+OBJECT_SCHEMA_NAME(@TableId)+'.'+OBJECT_NAME(@TableId)+'('+@SQLReferences+')'

;WITH FKCandidate
AS (
	SELECT 
		C.object_id, 
		C.name,
		StandardisedObjectName= REPLACE(CASE WHEN LEFT(O.name,3)='tbl' THEN SUBSTRING(O.name,4,len(O.name)) ELSE O.name END,'_',''),
		StandardisedColumnName=REPLACE(C.name,'_','')
	FROM sys.columns AS C
		INNER JOIN sys.objects AS O
		ON C.object_id = O.object_id
	WHERE REPLACE(C.name,'_','') in (SELECT StandardisedColumnName FROM CandidateKey.StandardisedPKItems WHERE object_id=@TableID) -- Primary key fields
	AND C.object_id<>@TableId
	AND O.type='U'
	AND OBJECT_SCHEMA_NAME(O.object_id) NOT IN ('dba','validate','CandidateKey','sys')
	AND O.name NOT LIKE 'sysdiagram%'
),
FKQualifier AS 
(SELECT object_id
FROM FKCandidate
GROUP BY object_id
HAVING COUNT(*)=@PKFieldCount -- Identify the table objects that have all the fields in the primary key.
),
PKFieldOrder AS (
	SELECT	StandardisedColumnName, ColumnID
	FROM CandidateKey.StandardisedPKItems
	WHERE Object_Id = @TableID
)
INSERT INTO @FKTable(Object_Id,ColumnID,StandardisedObjectName,OriginalColumnName,StandardisedColumnName)
SELECT 
	FKCandidate.object_id,
	PKFieldOrder.ColumnID,
	FKCandidate.StandardisedObjectName,
	FKCandidate.name,
	FKCandidate.StandardisedColumnName
FROM FKQualifier INNER JOIN FKCandidate
 ON FKQualifier.object_id = FKCandidate.object_id
 INNER JOIN   PKFieldOrder
 ON PKFieldOrder.StandardisedColumnName = FKCandidate.StandardisedColumnName
ORDER BY PKFieldOrder.ColumnId


DECLARE @NextObject INT=0 -- Child table object_id
DECLARE @ForeignKeyText VARCHAR(MAX) -- Full blown ALTER TABLE ... FOREIGN KEY... REFERENCES statement.
DECLARE @ForeignKeyFields VARCHAR(MAX) -- delimited list of foreign key fields

WHILE @NextObject IS NOT NULL
	BEGIN
		SELECT @NextObject = MIN(Object_Id)
		FROM @FKTable
		WHERE Object_Id>@NextObject

		IF @NextObject IS NOT NULL
			BEGIN
				IF NOT EXISTS(
					SELECT FK.StandardisedColumnName 
					FROM @FKTable AS FK 
						INNER JOIN CandidateKey.GetFieldsUsedInExistingFK(@NextObject) AS CK
						ON FK.StandardisedColumnName = CK.StandardisedColumnName
					)
					BEGIN
						SET @ForeignKeyText='ALTER TABLE ' 
							+ OBJECT_SCHEMA_NAME(@NextObject)+'.'+OBJECT_NAME(@NextObject)
							+ ' WITH NOCHECK ADD CONSTRAINT FKCandidate_'+OBJECT_SCHEMA_NAME(@NextObject)+'_'+OBJECT_NAME(@NextObject) 
							+ '_' +OBJECT_SCHEMA_NAME(@TableId)+'_'+OBJECT_NAME(@TableId)
							+ ' FOREIGN KEY ('

						-- Build concatenated list of foreign key names
						SELECT @ForeignKeyFields = COALESCE(@ForeignKeyFields+',','')+OriginalColumnName
						FROM @FKTable
						WHERE Object_Id=@NextObject
						ORDER BY ColumnID

						SET @ForeignKeyText=@ForeignKeyText 
							+ @ForeignKeyFields + ') '
							+ @SQLReferences

						PRINT @ForeignKeyText

						BEGIN TRY
							EXEC(@ForeignKeyText)
						END TRY
						BEGIN CATCH
							PRINT '================================='
							RAISERROR('ERROR TableID=%i, SQL=%s',10,1,@TableID,@ForeignKeyText) WITH NOWAIT
							PRINT '================================='
						END CATCH
						SET @ForeignKeyFields = NULL
					END
				END
	END
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GetCandidateForeignKey'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GetStandardisedPKItems
AS
SET NOCOUNT ON
TRUNCATE TABLE CandidateKey.StandardisedPKItems

INSERT INTO CandidateKey.StandardisedPKItems (
	Object_id, 
	ColumnId, 
	SchemaName, 
	OriginalObjectName, 
	StandardisedObjectName, 
	OriginalColumnName, 
	StandardisedColumnName)
SELECT 
	O.Object_Id,
	C.Column_Id,
	OBJECT_SCHEMA_NAME(c.object_id),
	O.name,
	StandardisedObjectName= REPLACE(CASE WHEN LEFT(O.name,3)='tbl' THEN SUBSTRING(O.name,4,len(O.name)) ELSE O.name END,'_',''),
	OriginalColumnName=C.name,
	StandardisedColumnName=REPLACE( -- Deal with primary key fields like Id & code and the tibblers.
		CASE	WHEN C.name IN('Id','Code') AND LEFT(O.name,3)='tbl' THEN SUBSTRING(O.name,4,len(O.name))+C.name
				WHEN C.name IN('Id','Code') AND LEFT(O.name,3)!='tbl' THEN O.name + c.name
				ELSE C.name END,
		'_','')
FROM sys.index_columns AS IC
	INNER JOIN sys.indexes AS I
	ON IC.object_id = I.object_id
	AND IC.index_id = I.index_id
	INNER JOIN sys.columns AS C
	ON IC.object_id = C.object_id
	AND IC.index_column_id = C.column_id
	INNER JOIN sys.objects AS O
	ON O.object_id=C.object_id
WHERE  I.is_primary_key=1
AND IC.is_included_column=0
AND OBJECT_SCHEMA_NAME(c.object_id) NOT IN ('Validate','CandidateKey','dba','sys') -- Exclude mechanical schemas
AND O.name NOT LIKE 'sysdiagram%'
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GetStandardisedPKItems'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GetPKOneToOneZeroPriority
AS
SET NOCOUNT ON

TRUNCATE TABLE CandidateKey.PKOneToOneZero


;WITH MultiField AS ( -- Single field primary keys that occur more than once i.e. 1 to 0..1 relationships.
	SELECT pk.StandardisedColumnName
	FROM CandidateKey.StandardisedPKItems AS PK -- Table to contain standardised names of primary key artefacts
		INNER JOIN CandidateKey.PKFieldCount AS PKC -- View to determine number of fields in a primary key.
		ON pk.Object_id = PKC.Object_id
	WHERE PKC.PKFieldCount=1
	GROUP BY PK.StandardisedColumnName
	HAVING COUNT(*)>1
)
INSERT INTO   CandidateKey.PKOneToOneZero(
	Object_Id, 
	TablePriority, 
	SchemaName, 
	TableName, 
	StandardisedTableName, 
	OriginalColumnName, 
	StandardisedColumnName
)
SELECT 
	PK.Object_id,
	TablePriority=CASE 
		WHEN PK.OriginalColumnName = 'ID' THEN 1
		WHEN PK.OriginalColumnName='Code' THEN 2
		WHEN PATINDEX(StandardisedObjectName+'%',PK.StandardisedColumnName)=1 THEN 3
		WHEN OBJECT_SCHEMA_NAME(PK.object_id) = object_name(PK.object_id) THEN 4
		ELSE 5
	END,
	SchemaName=OBJECT_SCHEMA_NAME(PK.object_id),
	TableName=object_name(PK.object_id),
	StandardisedObjectName,
	OriginalColumnName,
	StandardisedColumnName
FROM CandidateKey.StandardisedPKItems AS PK
	INNER JOIN CandidateKey.PKFieldCount AS PKC 
	ON PK.Object_id = PKC.Object_id
WHERE PK.StandardisedColumnName IN (select StandardisedColumnName from MultiField)
AND PKC.PKFieldCount=1
ORDER BY StandardisedColumnName
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GetPKOneToOneZeroPriority'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GeneratePKOneToOneZeroHierarchy
AS
SET NOCOUNT ON

DECLARE @ProcName VARCHAR(256)=OBJECT_SCHEMA_NAME(@@PROCID)+'.'+OBJECT_NAME(@@PROCID)
PRINT '==============================================='
RAISERROR('PROC: %s',10,1,@ProcName) WITH NOWAIT

DECLARE @SQL VARCHAR(MAX)
DECLARE @CRLF CHAR(2)=CHAR(13)+CHAR(10)

;WITH FKHierarchy AS (
	SELECT 
		FKSequence=ROW_NUMBER() OVER (PARTITION BY StandardisedColumnName ORDER BY SchemaName,TablePriority ),
		Object_Id, TablePriority, SchemaName, TableName, StandardisedTableName, OriginalColumnName, StandardisedColumnName 
	FROM CandidateKey.PKOneToOneZero
)
SELECT @SQL = COALESCE(@SQL+';'+@CRLF,'')
+	'ALTER TABLE '
+	QUOTENAME(c.SchemaName)+'.'+QUOTENAME(C.TableName)
+	' ADD CONSTRAINT '
+	QUOTENAME('FKCandidate_'+c.SchemaName+'_'+C.TableName+'_'+P.schemaName+'_'+P.tablename)
+	' FOREIGN KEY ('
+	C.originalColumnName
+	') REFERENCES '
+	QUOTENAME(P.SchemaName)+'.'+QUOTENAME(P.TableName)
+	'('
+	p.OriginalColumnName
+	')'

FROM FKHierarchy AS P
	INNER JOIN FKHierarchy AS C
	ON p.StandardisedColumnName = c.StandardisedColumnName
	AND P.Object_id<>c.Object_id
	AND P.SchemaName = C.SchemaName
	AND P.TablePriority<c.TablePriority
	AND P.FKSequence=C.FKSequence-1
	AND C.StandardisedColumnName NOT IN(SELECT StandardisedColumnName FROM CandidateKey.GetFieldsUsedInExistingFK( C.Object_id ))
ORDER BY P.TablePriority
PRINT @SQL
EXEC(@SQL)


GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GeneratePKOneToOneZeroHierarchy'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GenerateMultiFieldFK
AS
SET NOCOUNT ON

DECLARE @ProcName VARCHAR(256)=OBJECT_SCHEMA_NAME(@@PROCID)+'.'+OBJECT_NAME(@@PROCID)
RAISERROR('PROC: %s',10,1,@ProcName) WITH NOWAIT

DECLARE	
	@object_id INT, 
	@PKFieldCount TINYINT

DECLARE csr_FK INSENSITIVE CURSOR  
FOR SELECT 
	Object_Id,
	PKFieldCount
FROM  CandidateKey.PKFieldCount
WHERE PKFieldCount>1
ORDER BY PKFieldCount DESC

OPEN csr_FK

FETCH NEXT FROM csr_FK INTO @object_id,@PKFieldCount

WHILE @@FETCH_STATUS = 0
	BEGIN
		EXEC CandidateKey.GetCandidateForeignKey @object_id
		FETCH NEXT FROM csr_FK INTO @object_id,@PKFieldCount

	END

CLOSE csr_FK
DEALLOCATE csr_FK

GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GenerateMultiFieldFK'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GenerateSchemaSpecificFK
AS
SET NOCOUNT ON

DECLARE @ProcName VARCHAR(256)=OBJECT_SCHEMA_NAME(@@PROCID)+'.'+OBJECT_NAME(@@PROCID)
PRINT '==============================================='
RAISERROR('PROC: %s',10,1,@ProcName) WITH NOWAIT
 
DECLARE 
	@SQL VARCHAR(MAX),
	@CRLF CHAR(2)=CHAR(13)+CHAR(10)
SELECT @SQL=COALESCE(@SQL+';'+@CRLF,'')
+	'ALTER TABLE '
+	QUOTENAME(c.SchemaName)+'.'+QUOTENAME(C.OriginalObjectName)
+	' WITH NOCHECK ADD CONSTRAINT '
+	QUOTENAME('FKCandidate_'+c.SchemaName+'_'+C.OriginalObjectName+'_'+P.schemaName+'_'+P.OriginalObjectname)
+	' FOREIGN KEY ('
+	C.originalColumnName
+	') REFERENCES '
+	QUOTENAME(P.SchemaName)+'.'+QUOTENAME(P.OriginalObjectName)
+	'('
+	p.OriginalColumnName
+	')' 
FROM  CandidateKey.StandardisedPKItems AS P
		INNER JOIN 	CandidateKey.PKFieldCount AS FC
		ON P.Object_Id = FC.Object_Id
		AND FC.PKFieldCount=1
		INNER JOIN CandidateKey.StandardisedPKItems AS C
		ON P.SchemaName = C.SchemaName
		AND P.Object_Id <> C.Object_Id
		AND PATINDEX(P.StandardisedObjectName+'%',C.StandardisedObjectName)=1
		AND P.StandardisedColumnName = C.StandardisedColumnName
WHERE C.StandardisedColumnName NOT IN(SELECT StandardisedColumnName FROM CandidateKey.GetFieldsUsedInExistingFK( C.Object_id ))
PRINT @SQL
EXEC(@SQL)
GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GenerateSchemaSpecificFK'
GO
-------------------------------------------------------------------------------------------------
CREATE PROC CandidateKey.GenerateFinalSingleFieldFK
AS
SET NOCOUNT ON

DECLARE @ProcName VARCHAR(256)=OBJECT_SCHEMA_NAME(@@PROCID)+'.'+OBJECT_NAME(@@PROCID)
PRINT '==============================================='
RAISERROR('PROC: %s',10,1,@ProcName) WITH NOWAIT

DECLARE @SingleFieldPKs TABLE (Object_Id INT NOT NULL PRIMARY KEY CLUSTERED)

DECLARE @Object_Id INT=0

;with MO AS ( -- For the 1 to 0..1 relationships rank the priority
	SELECT
		Object_id,
		SequenceNumber=row_number() over (partition by StandardisedColumnName order by StandardisedColumnName, TablePriority),
		StandardisedColumnName
	from CandidateKey.PKOneToOneZero
),
PriorityObject AS ( -- Choose the master table in the 1 to 0..1 relationships
	SELECT StandardisedColumnName,Object_Id 
	FROM MO
	WHERE SequenceNumber=1
)
-- Exclude tables if they participate in a 1 to 0..1 relationship and are not the master.
INSERT INTO  @SingleFieldPKs(Object_Id)
SELECT FC.Object_Id
FROM CandidateKey.PKFieldCount FC
	INNER JOIN CandidateKey.StandardisedPKItemS AS SPK
	ON FC.Object_Id = SPK.Object_Id
	LEFT JOIN PriorityObject
	ON SPK.StandardisedColumnName =  PriorityObject.StandardisedColumnName
WHERE FC.PKFieldCount=1
AND FC.Object_Id = ISNULL(PriorityObject.Object_Id,FC.Object_ID)


WHILE @Object_Id IS NOT NULL
	BEGIN
		SELECT @Object_Id = MIN(Object_Id)
		FROM @SingleFieldPKs
		WHERE Object_Id>@Object_Id

		IF @Object_Id IS NOT NULL
			EXEC CandidateKey.GetCandidateForeignKey @object_id
	END

GO
IF @@ERROR=0
	PRINT 'PROC CREATED: CandidateKey.GenerateFinalSingleFieldFK'
GO
-------------------------------------------------------------------------------------------------
