USE [Monitordb]
GO

IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE name = 'Check_Backups' and type = 'P')
EXEC ('CREATE PROC [dbo].[Check_Backups] AS PRINT ''STUB VERSION'' ')
GO

ALTER PROC [dbo].[Check_Backups]
      @Server_Name SYSNAME = @@SERVERNAME
    , @days_back   TINYINT = 7     -- # of days for backup history retrieval.                                           Default = 7
    , @min_fulls   TINYINT = 1     -- min. # of days with full backup required in period defined by @days_back.         Default = 1
    , @min_diffs   TINYINT = 6     -- min. # of days with differential backup required in period defined by @days_back. Default = 6
    , @min_logs    TINYINT = 7     -- min. # of days with log backup required in period defined by @days_back.          Default = 7
AS
  /*   Record backup information from a remote server in a holding table on the monitor server for reporting/alerting.
       Steps:
       1) Per server that is queried, record backup details in a table variable ('@backups');
       2) Cleanup monitor server holding table ('backup_check') for server being queried;
       3) Update monitor server holding table with info recorded in step 1 and include some logic to print warnings, based on @min_x params specified.
       
       Follow up:
       Email alerts can be sent through the SP 'Mail_Results_Check_Backups'. This SP will mail any entries marked by the word 'Check' in a tabular format.
       
       Exclusions:
       Databases can be excluded from this check by entering the database name in the table msdb.dbo.ExcludeFromMaintenance on the linked server 
       (column name 'DatabaseName', type SYSNAME)
  */
  
  -- Determine whether exclusion table exists on linked server (@exclude_table=1) or not (@exclude_table=0)
  -- Uses a global temp table to store a value and pass on to a variable; improvements are welcome!
SET NOCOUNT ON   
CREATE TABLE ##CheckBackupsTmp ([Exists] BIT)
INSERT ##CheckBackupsTmp EXEC ('SELECT CASE WHEN (SELECT [id] FROM ['+@Server_Name+'].msdb.dbo.sysobjects WHERE [name] = ''ExcludeFromMaintenance'' AND [type] =''U'')> 0 THEN 1 ELSE 0 END')
DECLARE @exclude_table BIT
SELECT  @exclude_table = [Exists] FROM ##CheckBackupsTmp
DROP TABLE ##CheckBackupsTmp                         

  -- Build the SQL command string. 
DECLARE @cmd NVARCHAR(MAX)
SELECT  @cmd = N''
SELECT  @cmd = @cmd + '
 SET NOCOUNT ON  -- Record basic database device and backup info in a temp table
 DECLARE @backups TABLE (
     [database_name]   SYSNAME
   , [type]            CHAR(1)
   , [last_backup]     DATETIME
   , [count]           SMALLINT
   , [backup_days]     INT
   )
 
 INSERT @backups
  SELECT   bs.[database_name]
         , bs.[type]
         , MAX(bs.backup_finish_date)
         , COUNT(*) 
         , COUNT(DISTINCT DATEPART(DY,bs.backup_finish_date))  -- # of distinct days on which at least one backup was made, per backup type (to check against the minimum number of daily backups required)
  FROM     [' + @Server_Name + '].msdb.dbo.backupset bs
  INNER JOIN 
           [' + @Server_Name + '].msdb.dbo.backupmediafamily bmf
  ON       bs.media_set_id = bmf.media_set_id 
  WHERE    bs.backup_start_date > DATEADD(day,DATEDIFF(day,0,GETDATE()),0)-' + CONVERT(VARCHAR(3),@days_back) + ' -- 00:00 at date specified by @days_back
  AND      bs.backup_start_date < DATEADD(day,DATEDIFF(day,0,GETDATE()),0)                                        -- 00:00 today
  GROUP BY bs.database_name
         , bs.[type]

 DELETE backup_check WHERE server_name = ''' + @Server_Name + '''  -- Delete old info from monitor server holding table
 INSERT INTO backup_check     -- Update monitor server holding table
 SELECT  DISTINCT 
         ''' + @Server_Name + '''
       , d.name
       , d.create_date
       , d.recovery_model
       , (SELECT [count] FROM @backups WHERE type = ''D'' AND database_name = d.name)             -- Record # of full backups during sampling interval
       , (SELECT [count] FROM @backups WHERE type = ''I'' AND database_name = d.name)             -- Record # of differential backups during sampling interval
       , CASE 
           WHEN d.recovery_model != 3 
           THEN (SELECT [count] FROM @backups WHERE type = ''L'' AND database_name = d.name)      -- Record # of log backups during sampling interval
           ELSE -1                                                                                -- or print negative number to indicate recovery model = simple
         END
       , (SELECT MAX([last_backup]) FROM @backups WHERE type = ''D'' AND database_name = d.name)  -- Record date/time of last full backup
       , (SELECT MAX([last_backup]) FROM @backups WHERE type = ''I'' AND database_name = d.name)  -- Record date/time of last diff backup
       , (SELECT MAX([last_backup]) FROM @backups WHERE type = ''L'' AND database_name = d.name)  -- Record date/time of last log backup
       , CASE 
           WHEN ISNULL((SELECT [backup_days] FROM @backups WHERE type = ''D'' AND database_name = d.name),0) < '+ CONVERT(VARCHAR(3),@min_fulls) + ' 
           THEN ISNULL((SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''D'' AND database_name = d.name),''0'') + ''(!)''
           ELSE (SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''D'' AND database_name = d.name) 
         END       -- Print # of days with at least one full backup + warning if # of days with a full backup is below @min_fulls.
       , CASE 
           WHEN ISNULL((SELECT [backup_days] FROM @backups WHERE type = ''I'' AND database_name = d.name),0) < '+ CONVERT(VARCHAR(3),@min_diffs) + '
           THEN ISNULL((SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''I'' AND database_name = d.name),''0'') + ''(!)''
           ELSE (SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''I'' AND database_name = d.name)
         END       -- Print # of days with at least one diff backup + warning if # of days with a diff backup is below @min_diff.
       , CASE
           WHEN d.recovery_model != 3   -- if recovery is not simple
            AND ISNULL((SELECT [backup_days] FROM @backups WHERE type = ''L'' AND database_name = d.name),0) < '+ CONVERT(VARCHAR(3),@min_logs) + '
           THEN ISNULL((SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''L'' AND database_name = d.name),''0'') + ''(!)''
           ELSE (SELECT CONVERT(VARCHAR(6),[backup_days]) FROM @backups WHERE type = ''L'' AND database_name = d.name)
         END       -- Print # of days with at least one log backup + warning if # of days with a log backup is below @min_logs.
 FROM    [' + @Server_Name + '].master.sys.databases d
 LEFT OUTER JOIN  @backups b
 ON      b.database_name = d.name
 WHERE   d.state = 0                              -- online databases only  
 AND     d.name NOT IN (''model'',''tempdb'')     -- exclude certain system dbs
'
 -- Extend query to exclude databases in local exclusion table, if that exists
SELECT @cmd = @cmd
               + CASE WHEN @exclude_table = 0 
                      THEN '' 
                      ELSE '
 AND     d.name NOT IN                            -- do not report if DB exists in exclusion table
               (SELECT DatabaseName               
                FROM   [' + @Server_Name + '].[msdb].dbo.[ExcludeFromMaintenance] ) -- exclude database(s) in local exclusion table'
                 END

-- Execute query and store results in holding table
EXEC sp_executesql @cmd

GO
